/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/qt/variant.h>

namespace
{
typedef QPair<QVariant, QVariant> QVariantPair;
static const QVariant::Type qpair_type_id{
    static_cast<QVariant::Type>(qRegisterMetaType<QVariantPair>("QPair<QVariant,QVariant>"))};
}

messaging::qt::Variant::Variant(const QVariant& qv)
    : qv{qv}
{
}

messaging::qt::Variant::Variant(const messaging::qt::Variant& rhs)
    : qv{rhs.qv}
{
}

messaging::qt::Variant& messaging::qt::Variant::operator=(const messaging::qt::Variant& rhs)
{
    qv = rhs.qv;
    return *this;
}

messaging::Variant::Type messaging::qt::Variant::type() const
{
    switch (qv.type())
    {
        case QVariant::UInt:
        case QVariant::Int:
        case QVariant::ULongLong:
        case QVariant::LongLong:
            return messaging::Variant::Type::integer;
        case QVariant::Bool:
            return messaging::Variant::Type::boolean;
        case QVariant::Double:
            return messaging::Variant::Type::floating_point;
        case QVariant::String:
            return messaging::Variant::Type::string;
        case QVariant::Map:
        case QVariant::List:
            return messaging::Variant::Type::recursive;
        case QVariant::ByteArray:
            return messaging::Variant::Type::data;
        default:
        {
            // Unfortunately, qpair_type_id is not a compile-time constant.
            if (qv.type() == qpair_type_id)
            {
                return messaging::Variant::Type::recursive;
            }

            throw std::runtime_error{"Could not map type."};
        }
    }

    throw std::logic_error{"Should not reach here."};
}

bool messaging::qt::Variant::as_bool() const
{
    Variant::throw_if_type_mismatch<messaging::Variant::Type::boolean>(type());
    return qv.toBool();
}

std::int64_t messaging::qt::Variant::as_int() const
{
    Variant::throw_if_type_mismatch<messaging::Variant::Type::integer>(type());
    return qv.toInt();
}

double messaging::qt::Variant::as_double() const
{
    Variant::throw_if_type_mismatch<messaging::Variant::Type::floating_point>(type());
    return qv.toDouble();
}

std::string messaging::qt::Variant::as_string() const
{
    Variant::throw_if_type_mismatch<messaging::Variant::Type::string>(type());
    return qv.toString().toStdString();
}

const char* messaging::qt::Variant::as_data() const
{
    Variant::throw_if_type_mismatch<messaging::Variant::Type::data>(type());
    return qv.toByteArray();
}

std::size_t messaging::qt::Variant::data_size() const
{
    Variant::throw_if_type_mismatch<messaging::Variant::Type::data>(type());
    return qv.toByteArray().size();
}

std::size_t messaging::qt::Variant::size() const
{
    Variant::throw_if_type_mismatch<messaging::Variant::Type::recursive>(type());

    switch (qv.type())
    {
        case QVariant::Map:
            return qv.toMap().size();
        case QVariant::List:
            return qv.toList().size();
        default:
            // Unfortunately, qpair_type_id is not a compile-time constant.
            if (qv.type() == qpair_type_id)
            {
                return 2;
            }

            break;
    }

    throw std::logic_error{"Should not reach here"};
}

std::deque<std::string> messaging::qt::Variant::keys() const
{
    Variant::throw_if_type_mismatch<messaging::Variant::Type::recursive>(type());

    switch (qv.type())
    {
        case QVariant::Map: {
            std::deque<std::string> keysArray;
            QVariantMap map = qv.toMap();
            Q_FOREACH(QString key, map.keys()) {
                keysArray.push_back(key.toStdString());
            }
            return keysArray;
        }
        default:
            throw std::logic_error{"Variant.keys() operation only allowed on maps"};
    }
}

std::shared_ptr<messaging::Variant> messaging::qt::Variant::at(const std::string& key) const
{
    Variant::throw_if_type_mismatch<messaging::Variant::Type::recursive>(type());

    switch (qv.type())
    {
        case QVariant::Map:
        {
            QVariantMap map = qv.toMap();
            const QVariant v = map.value(QString::fromStdString(key));
            return std::make_shared<qt::Variant>(v);
        }
        default:
            throw std::logic_error{"Variant.value(key) operation only allowed on maps"};
    }
}

std::shared_ptr<messaging::Variant> messaging::qt::Variant::at(std::size_t idx) const
{
    Variant::throw_if_type_mismatch<messaging::Variant::Type::recursive>(type());

    switch (qv.type())
    {
        case QVariant::Map:
        {
            // Kind of ugly, but Qt only returns by value for accessors,
            // instead of providing immutable references. With that, we have to
            // make sure that the map stays alive while we are using the iterator
            // into it.
            auto m = qv.toMap();
            auto it = m.begin() + idx;
            return std::make_shared<qt::Variant>(QVariant::fromValue(qMakePair(QVariant(it.key()), it.value())));
        }
        case QVariant::List:
            return std::make_shared<qt::Variant>(qv.toList().at(idx));
        default:
            if (qv.type() == qpair_type_id)
            {
                if (idx > 1)
                    throw std::out_of_range{"Pair only contains two elements"};

                return std::make_shared<qt::Variant>(
                    (idx == 0 ? qv.value<QVariantPair>().first : qv.value<QVariantPair>().second));
            }

            break;
    }

    throw std::logic_error{"Should not reach here"};
}
