/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <TelepathyQt/DBusObject>
#include <messaging/qt/tp/interfaces/base_channel_roles.h>
#include <messaging/qt/tp/interfaces/base_channel_roles_internal.h>
#include <messaging/qt/tp/interfaces/constants.h>
#include <messaging/qt/tp/interfaces/types.h>

namespace mqt = messaging::qt::tp;
namespace mqti = messaging::qt::tp::interfaces;

// Chan.I.Roles
struct MESSAGING_FW_LOCAL mqti::BaseChannelRolesInterface::Private {
    Private(BaseChannelRolesInterface *parent)
        : canUpdateRoles(false)
        , adaptee(new BaseChannelRolesInterface::Adaptee(parent)) {
    }

    void emitRolesChangedSignal(const HandleRolesMap &added, const HandleRolesMap &removed) const;

    HandleRolesMap roles;
    bool canUpdateRoles;
    UpdateRolesCallback updateRolesCB;
    BaseChannelRolesInterface::Adaptee *adaptee;
};

void mqti::BaseChannelRolesInterface::Private::emitRolesChangedSignal(const HandleRolesMap &added, const HandleRolesMap &removed) const
{
    QMetaObject::invokeMethod(adaptee, "rolesChanged",
                              Q_ARG(HandleRolesMap, added),
                              Q_ARG(HandleRolesMap, removed));
}

mqti::BaseChannelRolesInterface::Adaptee::Adaptee(BaseChannelRolesInterface *interface)
    : QObject(interface),
      mInterface(interface)
{
    qDBusRegisterMetaType<HandleRolesMap>();
    // this signal connections and the exposition of the rolesChanged signal is only needed for unit tests.
    // In case unit testing is changed this could be removed and not expose signal in the interface
    qRegisterMetaType<HandleRolesMap>("HandleRolesMap");
    connect(this, SIGNAL(rolesChanged(const HandleRolesMap &, const HandleRolesMap&)),
                     mInterface, SIGNAL(rolesChanged(const HandleRolesMap &, const HandleRolesMap&)));
}

mqti::BaseChannelRolesInterface::Adaptee::~Adaptee()
{
}

mqti::HandleRolesMap mqti::BaseChannelRolesInterface::Adaptee::roles() const
{
    return mInterface->roles();
}

bool mqti::BaseChannelRolesInterface::Adaptee::canUpdateRoles() const
{
    return mInterface->canUpdateRoles();
}

void mqti::BaseChannelRolesInterface::Adaptee::updateRoles(const HandleRolesMap &contactRoles,
                                                           const ChannelInterfaceRolesAdaptor::UpdateRolesContextPtr &context)
{
    if (!mInterface->mPriv->updateRolesCB.isValid()) {
        context->setFinishedWithError(TP_QT_ERROR_NOT_IMPLEMENTED, QLatin1String("Not implemented"));
        return;
    }

    if (!mInterface->canUpdateRoles()) {
        context->setFinishedWithError(TP_QT_ERROR_PERMISSION_DENIED, QLatin1String("Operation not allowed; anUpdateRoles flag set to false"));
        return;
    }

    Tp::DBusError error;
    mInterface->mPriv->updateRolesCB(contactRoles, &error);
    if (error.isValid()) {
        context->setFinishedWithError(error.name(), error.message());
        return;
    }
    context->setFinished();
}

/**
 * \class BaseChannelRolesInterface
 *
 * \brief Base class for implementations of Channel.Interface.Roles
 *
 */

/**
 * Class constructor.
 */
mqti::BaseChannelRolesInterface::BaseChannelRolesInterface()
    : Tp::AbstractChannelInterface(TP_QT_IFACE_CHANNEL_INTERFACE_ROLES),
      mPriv(new Private(this))
{
}

/**
 * Class destructor.
 */
mqti::BaseChannelRolesInterface::~BaseChannelRolesInterface()
{
    delete mPriv;
}

void mqti::BaseChannelRolesInterface::setUpdateRolesCallback(const UpdateRolesCallback &cb)
{
    mPriv->updateRolesCB = cb;
}

/**
 * Return the immutable properties of this interface.
 *
 * Immutable properties cannot change after the interface has been registered
 * on a service on the bus with registerInterface().
 *
 * \return The immutable properties of this interface.
 */
QVariantMap mqti::BaseChannelRolesInterface::immutableProperties() const
{
    QVariantMap map;
    return map;
}

void mqti::BaseChannelRolesInterface::createAdaptor()
{
    (void) new ChannelInterfaceRolesAdaptor(dbusObject()->dbusConnection(),
            mPriv->adaptee, dbusObject());
}

void mqti::BaseChannelRolesInterface::updateRoles(const HandleRolesMap &contactRoles, Tp::DBusError *error)
{
    if (!mPriv->updateRolesCB.isValid()) {
        error->set(TP_QT_ERROR_NOT_IMPLEMENTED, QLatin1String("Not implemented"));
        return;
    }

    if (!canUpdateRoles()) {
        error->set(TP_QT_ERROR_PERMISSION_DENIED, QLatin1String("Operation not allowed; canUpdateRoles flag set to false"));
    }

    return mPriv->updateRolesCB(contactRoles, error);
}

mqti::HandleRolesMap mqti::BaseChannelRolesInterface::roles() const
{
    return mPriv->roles;
}

void mqti::BaseChannelRolesInterface::setRoles(const HandleRolesMap &receivedMap)
{
    HandleRolesMap addedMap;
    HandleRolesMap removedMap;

    Q_FOREACH (uint handle, receivedMap.keys())
    {
        const ChannelRoles received(receivedMap.value(handle));
        const ChannelRoles current(mPriv->roles.value(handle));

        const ChannelRoles kept     = current & received;
        const ChannelRoles added    = received & ~kept;
        const ChannelRoles removed  = current & ~kept;

        if (added != 0) {
            addedMap[handle] = added;
        }

        if (removed != 0) {
            removedMap[handle] = removed;
        }

        // as this handle is already processed, remove it from current roles map
        mPriv->roles.remove(handle);
    }

    // remaining entries in current roles map are not included in received one, so add them to removed
    removedMap.unite(mPriv->roles);

    mPriv->roles = receivedMap;

    // emit RolesChanged only if there is any effective change
    if (!addedMap.empty() || !removedMap.empty()) {
        mPriv->emitRolesChangedSignal(addedMap, removedMap);
    }
}

bool mqti::BaseChannelRolesInterface::canUpdateRoles() const
{
    return mPriv->canUpdateRoles;
}

void mqti::BaseChannelRolesInterface::setCanUpdateRoles(bool canUpdateRoles)
{
    mPriv->canUpdateRoles = canUpdateRoles;
    notifyPropertyChanged(QLatin1String("CanUpdateRoles"), QVariant::fromValue(canUpdateRoles));
}
