#include <messaging/qt/network_monitor.h>

#include <messaging/connection.h>

#include <QtNetwork/QNetworkConfigurationManager>

#include <glog/logging.h>

namespace mq = messaging::qt;

mq::NetworkMonitor::NetworkMonitor(const std::weak_ptr<Connection> &connection, QObject *parent)
    : QObject(parent)
    , network_config_manager_{this}
    , connection_{connection}
    , current_ssid_{}
    , is_connection_ready_{false}
{
    start_monitoring();
}

mq::NetworkMonitor::~NetworkMonitor()
{
}

/*!
 * \brief starts monitoring current connection. The monitoring will restart in case a new
 * connection is created but this one won't be monitored anymore
 */
void mq::NetworkMonitor::start_monitoring()
{
    connect(&network_config_manager_,
            SIGNAL(onlineStateChanged(bool)),
            SLOT(refresh()), Qt::QueuedConnection);
    connect(&network_config_manager_,
            SIGNAL(configurationAdded(QNetworkConfiguration)),
            SLOT(refresh()), Qt::QueuedConnection);
    connect(&network_config_manager_,
            SIGNAL(configurationChanged(QNetworkConfiguration)),
            SLOT(refresh()), Qt::QueuedConnection);
    connect(&network_config_manager_,
            SIGNAL(configurationRemoved(QNetworkConfiguration)),
            SLOT(refresh()), Qt::QueuedConnection);
    connect(&network_config_manager_,
            SIGNAL(updateCompleted()),
            SLOT(refresh()), Qt::QueuedConnection);

    refresh_timer_.setSingleShot(true);
    connect(&refresh_timer_,
            SIGNAL(timeout()),
            SLOT(idle_refresh()), Qt::QueuedConnection);
}

/*!
 * \brief stops monitoring network switch in this connection
 */
void mq::NetworkMonitor::stop_monitoring()
{
    disconnect(&network_config_manager_,
            SIGNAL(onlineStateChanged(bool)),
            this,
            SLOT(refresh()));
    disconnect(&network_config_manager_,
            SIGNAL(configurationAdded(QNetworkConfiguration)),
            this,
            SLOT(refresh()));
    disconnect(&network_config_manager_,
            SIGNAL(configurationChanged(QNetworkConfiguration)),
            this,
            SLOT(refresh()));
    disconnect(&network_config_manager_,
            SIGNAL(configurationRemoved(QNetworkConfiguration)),
            this,
            SLOT(refresh()));
    disconnect(&network_config_manager_,
            SIGNAL(updateCompleted()),
            this,
            SLOT(refresh()));

    current_ssid_ = std::string{};
}

/*!
 * \brief flag needed to know when the connection object has become effectively online.
 * This is needed to determinate if a received ssid different from the previous one should be taken
 * as the trigger
 */
void mq::NetworkMonitor::set_connection_ready(bool ready)
{
    is_connection_ready_ = ready;
}

void mq::NetworkMonitor::refresh()
{
    refresh_timer_.start(3000);
}

void mq::NetworkMonitor::idle_refresh()
{
    // Check if is online
    QList<QNetworkConfiguration> active_configs = network_config_manager_.allConfigurations(QNetworkConfiguration::Active);
    bool is_online = active_configs.size() > 0;
    if (is_online) {
        // Check if the connection is wifi or ethernet
        QNetworkConfiguration default_config = network_config_manager_.defaultConfiguration();

        // while connecting, we admit updating over an empty current_ssid. In the case
        // of being effectively connected, that means a network switch
        if (not is_connection_ready_)
        {
            if (current_ssid_.empty())
            {
                dump(default_config);
                current_ssid_ = default_config.name().toStdString();
            }
        }

        if (current_ssid_ != default_config.name().toStdString())
        {
            // network is switched at this point, so desconnect the registered connection in case is valid.
            LOG(INFO) << "detected network switch";
            auto sp = connection_.lock();
            if (sp)
            {
                // disconnect, not specifying reason for fw try to reconnect when changed to a no connection status
                sp->disconnect(Connection::StatusChangedReason::network_error);
                stop_monitoring();
            }
        }

    } else {
        LOG(INFO) << "Network is offline";
    }
}

void mq::NetworkMonitor::dump(const QNetworkConfiguration &config)
{
    LOG(INFO) << "New network connection:\nType: " << config.bearerTypeName().toStdString()
              << "\nId: " << config.identifier().toStdString()
              << "\nName: " << config.name().toStdString()
              << "\nIsValid: " << config.isValid()
              << "\nRoamingAvailable: " << config.isRoamingAvailable();
}
