/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/group.h>

#include <messaging/qt/variant.h>

const std::string messaging::Group::ROOM_NAME = "roomName";
const std::string messaging::Group::DESCRIPTION = "description";
const std::string messaging::Group::SERVER = "server";
const std::string messaging::Group::CREATION_TIMESTAMP = "creationTimestamp";
const std::string messaging::Group::SUBJECT = "subject";

struct messaging::Group::Private
{
    messaging::Members initial_invitees;
    std::string initial_title;
    std::shared_ptr<messaging::Member> creator;
    messaging::VariantMap additional_properties;
};

messaging::Group::Group(std::string id, const messaging::Members &initial_invitees, std::string initial_title, const std::shared_ptr<Member> &creator, const VariantMap &additional_properties)
        : Recipient{id}, impl{new Private{initial_invitees, initial_title, creator, additional_properties}}
{
}

messaging::RecipientType messaging::Group::type() const
{
    return messaging::RecipientType::group;
}

messaging::Members messaging::Group::initial_invitees() const
{
    return impl->initial_invitees;
}

std::string messaging::Group::initial_title() const
{
    return impl->initial_title;
}

std::shared_ptr<messaging::Member> messaging::Group::creator() const
{
    return impl->creator;
}

messaging::VariantMap &messaging::Group::additional_properties()
{
    return impl->additional_properties;
}

std::shared_ptr<messaging::Variant> messaging::Group::additional_property(const std::string &key) const
{
    auto it = impl->additional_properties.find(key);
    if (it != impl->additional_properties.end()) {
        return it->second;
    }
    return std::make_shared<messaging::qt::Variant>(QVariant{});
}

void messaging::Group::set_additional_property(const std::string &key, const std::shared_ptr<messaging::Variant> &value)
{
    impl->additional_properties[key] = value;
}

bool messaging::Group::exists_additional_property(const std::string &key) const
{
    auto it = impl->additional_properties.find(key);
    return it != impl->additional_properties.end();
}
