/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <messaging/boost/variant.h>

#include <boost/format.hpp>

namespace
{
enum Idx
{
    boolean,
    int64_t,
    double_,
    string,
    data,
    vector,
    map
};

template <messaging::Variant::Type expected>
void throw_if_neq(messaging::Variant::Type actual)
{
    if (actual == expected)
    {
        return;
    }

    throw messaging::Variant::TypeMismatch(expected, actual);
}

boost::format type_mismatch_format()
{
    return boost::format("Type mismatch encountered - expected: %1%, actual: %2%");
}
}

messaging::BoostVariant::TypeMismatch::TypeMismatch(Type expected, Type actual)
    : std::runtime_error((type_mismatch_format() % expected % actual).str())
{
}

messaging::BoostVariant::BoostVariant(const messaging::BoostVariant::Value& v)
    : value(v)
{
}

messaging::BoostVariant::BoostVariant(bool b)
    : value(b)
{
}

messaging::BoostVariant::BoostVariant(std::int64_t i)
    : value(i)
{
}

messaging::BoostVariant::BoostVariant(double d)
    : value(d)
{
}

messaging::BoostVariant::BoostVariant(const std::string& s)
    : value(s)
{
}

messaging::BoostVariant::BoostVariant(const char* data, std::size_t size)
    : value(std::vector<char>{data, data + size})
{
}

messaging::BoostVariant::BoostVariant(const std::vector<char>& d)
    : value(d)
{
}

messaging::BoostVariant::BoostVariant(const std::vector<Value>& v)
    : value(v)
{
}

messaging::BoostVariant::BoostVariant(const std::map<Value, Value>& m)
    : value(m)
{
}

messaging::BoostVariant::BoostVariant(const messaging::BoostVariant& rhs)
    : value{rhs.value}
{
}

messaging::BoostVariant& messaging::BoostVariant::operator=(const messaging::BoostVariant& rhs)
{
    value = rhs.value;
    return *this;
}

messaging::Variant::Type messaging::BoostVariant::type() const
{
    switch (value.which())
    {
        case Idx::boolean:
            return Type::boolean;
        case Idx::int64_t:
            return Type::integer;
        case Idx::double_:
            return Type::floating_point;
        case Idx::string:
            return Type::string;
        case Idx::data:
            return Type::data;
        case Idx::vector:
        case Idx::map:
            return Type::recursive;
    }

    throw std::logic_error{"Should never reach here."};
}

bool messaging::BoostVariant::as_bool() const
{
    throw_if_neq<Type::boolean>(type());
    return boost::get<bool>(value);
}

std::int64_t messaging::BoostVariant::as_int() const
{
    throw_if_neq<Type::integer>(type());
    return boost::get<std::int64_t>(value);
}

double messaging::BoostVariant::as_double() const
{
    throw_if_neq<Type::floating_point>(type());
    return boost::get<double>(value);
}

std::string messaging::BoostVariant::as_string() const
{
    throw_if_neq<Type::string>(type());
    return boost::get<std::string>(value);
}

const char *messaging::BoostVariant::as_data() const
{
    throw_if_neq<Type::data>(type());
    return boost::get<std::vector<char>>(value).data();
}

std::size_t messaging::BoostVariant::data_size() const
{
    throw_if_neq<Type::data>(type());
    return boost::get<std::vector<char>>(value).size();
}

std::size_t messaging::BoostVariant::size() const
{
    throw_if_neq<Type::recursive>(type());
    switch (value.which())
    {
        case Idx::vector:
            return boost::get<std::vector<BoostVariant::Value>>(value).size();
        case Idx::map:
            return boost::get<std::map<BoostVariant::Value, BoostVariant::Value>>(value).size();
        default:
            break;
    }

    throw std::logic_error{"Should never reach here"};
}

std::deque<std::string> messaging::BoostVariant::keys() const
{
    throw_if_neq<Type::recursive>(type());

    switch (value.which())
    {
        case Idx::map: {
            std::deque<std::string> keys;
            auto m = boost::get<std::map<BoostVariant::Value, BoostVariant::Value>>(value);
            for (auto it = m.begin(); it != m.end(); it++) {
                auto k = std::make_shared<messaging::BoostVariant>(it->first);
                keys.push_back(k->as_string());
            }
            return keys;
        }
        default:
            throw std::logic_error{"Variant.keys() operation only allowed on maps"};
    }
}

std::shared_ptr<messaging::Variant> messaging::BoostVariant::at(const std::string &key) const
{
    throw_if_neq<Type::recursive>(type());

    switch (value.which())
    {
        case Idx::map: {
            auto m = boost::get<std::map<BoostVariant::Value, BoostVariant::Value>>(value);
            auto it = m.find(key);
            if (it != m.end()) {
                return std::make_shared<messaging::BoostVariant>(it->second);
            }
            throw std::out_of_range{"Could not find value for key " + key};
        }
        default:
            throw std::logic_error{"Variant.value(key) operation only allowed on maps"};
    }
}

std::shared_ptr<messaging::Variant> messaging::BoostVariant::at(std::size_t idx) const
{
    throw_if_neq<Type::recursive>(type());

    switch (value.which())
    {
        case Idx::vector:
            return std::make_shared<BoostVariant>(boost::get<std::vector<Value>>(value).at(idx));
        case Idx::map:
            break;  // return std::make_shared<BoostVariant>(boost::get<std::map<Value, Value>>().size();
        default:
            break;
    }

    throw std::logic_error{"Variant.at(index) operation only allowed for lists"};
}

/// @brief operator<< pretty-prints the given type to the given stream.
std::ostream& messaging::operator<<(std::ostream& out, messaging::Variant::Type type)
{
    switch (type)
    {
        case Variant::Type::boolean:
            return out << "boolean";
        case Variant::Type::integer:
            return out << "integer";
        case Variant::Type::floating_point:
            return out << "floating_point";
        case Variant::Type::string:
            return out << "string";
        case Variant::Type::recursive:
            return out << "recursive";
        case Variant::Type::data:
            // FIXME: check if we need to implement it
            return out;
    }

    return out;
}
