/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_RUNNER_H_
#define MESSAGING_RUNNER_H_

#include <QtDBus/QDBusConnection>

#include <messaging/visibility.h>

#include <messaging/qt/runtime.h>

#include <memory>
#include <string>

namespace messaging
{
// A handy forward declaration.
class Connector;
class ConnectorFactory;

/// @brief Runner provides a main method for executing a given messaging::Connector and
/// exposing it to telepathy as a connection manager instance.
struct MESSAGING_FW_PUBLIC Runner
{
    /// @brief Thrown if either help was explicitly requested or if a given configuration is
    /// incomplete/incorrect.
    struct MESSAGING_FW_PUBLIC HelpRequired : public std::runtime_error
    {
        HelpRequired(const std::string& help_text);
        std::string help_text;  ///< The help text that should be displayed to the user.
    };

    /// @brief All execution parameters go here.
    struct MESSAGING_FW_PUBLIC Configuration
    {
        /// @brief Parses a configuration from the given command line parameters.
        /// @throws HelpRequired if either help is explicitly requested or if the given arguments
        /// were incomplete/incorrect.
        MESSAGING_FW_PUBLIC static Configuration from_command_line_args(int argc,
                                                                        const char** argv,
                                                                        const ConnectorFactory& factory);

        int argc;
        const char** argv;                        ///< We pass on command line args;
        std::string connection_manager_name;      ///< The name for the connection manager.
        std::string protocol_name;                ///< The name of the protocol.
        std::shared_ptr<Connector> connector;     ///< The connector instance.
        std::shared_ptr<qt::Runtime> qt_runtime;  ///< The Qt runtime we operate under.
    };

    /// @brief Executes the runner with the given configuration.
    /// @throws std::exception in case of issues.
    MESSAGING_FW_PUBLIC static int main(const Configuration& configuration);
};
}

#endif  // MESSAGING_RUNNER_H_
