/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_RAII_H_
#define MESSAGING_RAII_H_

#include <functional>
#include <memory>
#include <thread>

namespace messaging
{
namespace raii
{
/// @brief AutoJoiningThread handles an underlying thread instance, joining it on destruction.
class AutoJoiningThread
{
public:
    /// @brief Constructs the empty instance, underlying thread is not running.
    AutoJoiningThread() = default;
    /// @brief Constructs a new instance, handing the given task to the underlying thread.
    AutoJoiningThread(std::function<void()> task)
        : worker(task)
    {
    }

    /// @brief Tries to join the underlying thread.
    ~AutoJoiningThread()
    {
        if (worker.joinable())
        {
            worker.join();
        }
    }

private:
    std::thread worker;
};

/// @brief Takes an instance which complies to the concept RunnableStopable
/// and executes it on a new thread, automatically stopping executing on
/// destruction, and joining the underlying thread.
///
/// RunnableStopable: We expect the type to provide a run method that
/// blocks for as long as an instance requires the thread of execution.
/// A call to stop should be non-blocking, requesting the instance
/// to shutdown operations.
template <typename RunnableStopable>
class Executor
{
public:
    /// @brief Creates a new instance, handing the given instance to the
    /// underlying thread.
    Executor(const std::shared_ptr<RunnableStopable>& rs)
        : rs(rs)
        , executor([this]()
                   {
                       this->rs->run();
                   })
    {
    }

    ~Executor()
    {
        rs->stop();
    }

private:
    std::shared_ptr<RunnableStopable> rs;  ///< The instance that should be executed.
    AutoJoiningThread executor;            ///< Auto-joining thread.
};
}
}

#endif  // MESSAGING_RAII_H_
