/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_PRESENCE_MANAGER_H
#define MESSAGING_PRESENCE_MANAGER_H

#include <messaging/visibility.h>
#include <messaging/non_copyable.h>
#include <messaging/non_movable.h>
#include <messaging/presence.h>

#include <memory>
#include <map>

namespace messaging
{
// A handy fw declaration
class Recipient;

/// @brief PresenceManager abstracts presence management
class MESSAGING_FW_PUBLIC PresenceManager : NonCopyable, NonMovable
{
  public:
    class Observer: NonCopyable, NonMovable
    {
      public:
        /// @brief on_presence_changed slot called when a presence status change has been produced in the observed object
        virtual void on_presence_changed(const std::shared_ptr<Recipient>& recipient, const Presence& presence) = 0;

      protected:
        /// @brief only inherited classes can created an observer
        Observer() = default;
    };

    /// @brief type defining a map of id <-> presence status spec, where id is a name like "offline", "available", etc..
    typedef std::map<std::string, StatusSpec> StatusMap;

    /// @brief gets the list of available statuses to be used with this plugin
    virtual StatusMap get_valid_statuses() = 0;

    /// @brief set_self_presence sets our status in the server
    virtual void set_self_presence(const std::string& status, const std::string& status_message = "") = 0;

    /// @brief request the presence of a recipient
    virtual void request_presence(const std::shared_ptr<Recipient>& recipient) = 0;

  protected:
    /// @brief Messenger constructs a new instance, installing the given observer instance.
    PresenceManager(const std::shared_ptr<Observer>& observer);

    //TODO(rmescandon): temporal signature; 'recipient' to be replaced with 'Target' object when available
    void announce_presence_changed(const std::shared_ptr<Recipient>& recipient, const Presence& presence);

  private:
    /// @cond
    struct Private;
    std::shared_ptr<Private> impl;
    /// @endcond
};

}

#endif  // MESSAGING_PRESENCE_MANAGER_H
