/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_CHAT_GROUP_MANAGER_H
#define MESSAGING_CHAT_GROUP_MANAGER_H

#include <messaging/visibility.h>
#include <messaging/non_copyable.h>
#include <messaging/non_movable.h>
#include <messaging/interface.h>
#include <messaging/member.h>
#include <messaging/members.h>
#include <messaging/flags.h>

#include <memory>
#include <chrono>

namespace messaging
{
// Maintain sequence of 2 multiples if adding new values to preserve flags management
enum class GroupPermissions : uint
{
    NoPermissions       = 0,      //0
    CanChangeTitle      = 1 << 0, //1
    CanKick             = 1 << 1, //2
    CanSetAdmin         = 1 << 2, //4
    CanDissolve         = 1 << 3, //8
    CanChangeSubject    = 1 << 4  //16
};

enum class CancelGroupReason
{
    None,       ///< No or unknown reason
    Error,      ///< Generic error makes this group temporally or permanent unavaliable for us
    Rejected,   ///< Group creation or rejoin was rejected
    Kicked,     ///< We were expelled from group
    Gone,       ///< Group is gone in server side and is not available anymore
    Leave       ///< We leave the group
};

/// @brief GroupManager models a textual group conversation.
class MESSAGING_FW_PUBLIC GroupManager : public Interface, NonCopyable, NonMovable
{
public:
    /// @brief Observer provides means to monitor the state of a GroupManager instance.
    class Observer : NonCopyable, NonMovable
    {
    public:
        virtual ~Observer() = default;

        virtual void on_group_created() = 0;
        virtual void on_group_cancelled(CancelGroupReason reason, const std::string &message) = 0;
        virtual void on_group_title_changed(const std::string& new_title) = 0;
        virtual void on_group_subject_changed(const std::string& new_subject,
                                            const std::shared_ptr<Member>& actor,
                                            const std::chrono::system_clock::time_point& when) = 0;
        virtual void on_members_updated(const Members& members) = 0;
        virtual void on_group_permissions_changed(const Flags<GroupPermissions> &permissions) = 0;

    protected:
        Observer() = default;
    };

    /// @brief leave_group exits group
    virtual void leave_group() = 0;

    /// @brief dissolve_group closes the current group and deletes the underlaying plugin conversation
    virtual void dissolve_group() = 0;

    /// @brief add_members adds an array of new users to the group. If the user alreay exists, this methods
    /// updates it in plugin. This can be used, for instance to change existing members roles.
    virtual void add_members(const Members& member) = 0;

    /// @brief remove_members removes the users in the array from the group
    virtual void remove_members(const Members& member) = 0;

    /// @brief change_group_title modifies the title of the group
    virtual void change_group_title(const std::string& title) = 0;

    /// @brief change_group_subject modifies the subject of the group
    virtual void change_group_subject(const std::string& subject) = 0;

    /// @brief group_id returns the id of the group being managed
    virtual std::string group_id() = 0;

    /// @brief group_title returns the title of the group being managed
    virtual std::string group_title() = 0;

    /// @brief group_subject returns the subject of the group being managed
    virtual std::string group_subject() = 0;

    /// @brief creator returns the creator of the group being managed
    virtual std::shared_ptr<Member> group_creator() = 0;

    /// @brief group_admins returns the list of admin identifiers
    virtual std::set<std::string> group_admins() = 0;

    /// @brief members returns current participants in the group.
    virtual Members members() = 0;

    /// @brief set_observer sets the observer to receive group manager callback notifications
    void set_observer(const std::shared_ptr<Observer> &observer);

    /// @brief permissions returns the permissions flags for this group and current user
    Flags<GroupPermissions> &permissions();

    /// @brief set_permissions sets the permissions flags for this group and current user
    void set_permissions(const Flags<GroupPermissions> &permissions);

protected:
    /// @brief GroupManager constructs a new instance.
    GroupManager();

    void validate_observer();

    void announce_group_created();
    void announce_group_cancelled(CancelGroupReason reason, const std::string& message);
    void announce_group_title_changed(const std::string& new_title);
    void announce_group_subject_changed(const std::string& new_subject,
                                        const std::shared_ptr<Member>& actor = std::make_shared<Member>(std::string{}),
                                        const std::chrono::system_clock::time_point& when = std::chrono::system_clock::now());
    void announce_members_updated(const Members& members);
    void announce_group_permissions_changed(const Flags<GroupPermissions> &permissions);

    /// @cond
    struct Private;
    std::shared_ptr<Private> impl;
    /// @endcond
};
}

#endif // MESSAGING_CHAT_GROUP_MANAGER_H
