/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_ENUMERATOR_H_
#define MESSAGING_ENUMERATOR_H_

#include <messaging/non_copyable.h>
#include <messaging/non_movable.h>

#include <functional>
#include <vector>

namespace messaging
{
/// @brief Enumerator<T> models enumeration of a collection of elements of type T.
template <typename T>
class Enumerator : NonCopyable, NonMovable
{
public:
    /// @brief enumerate invokes the given functor for all elements known to the Enumerator.
    virtual void enumerate(const std::function<void(const T&)>&) const = 0;

protected:
    // Only subclasses can instantiate.
    Enumerator() = default;
};

/// @brief StdVectorEnumerator implements Enumerator relying on a std::vector as backing store.
template <typename T>
class StdVectorEnumerator : public Enumerator<T>
{
public:
    StdVectorEnumerator(const std::vector<T>& v = std::vector<T>{})
        : v{v}
    {
    }

    void enumerate(const std::function<void(const T&)>& f) const override
    {
        for (const auto& element : v)
        {
            f(element);
        }
    }

    std::vector<T>& operator*()
    {
        return v;
    }

private:
    std::vector<T> v;
};
}

#endif  // MESSAGING_ENUMERATOR_H_
