/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MESSAGING_DICTIONARY_H_
#define MESSAGING_DICTIONARY_H_

#include <messaging/non_copyable.h>
#include <messaging/non_movable.h>

#include <string>

namespace messaging
{
/// @brief Dictionary models string values indexed by string keys.
template <typename Key, typename Value>
class Dictionary : NonCopyable, NonMovable
{
public:
    /// @brief operator[] is a handy overload dispatching to has_value_for_key.
    virtual const Value& operator[](const Key& key) const
    {
        return value_for_key(key);
    }

    /// @brief has_value_for_key returns true if the dictionary has a value for key.
    virtual bool has_value_for_key(const Key& key) const = 0;

    /// @brief value_for_key returns the value for the given key.
    /// @throw std::out_of_range if the key is not known to the dictionary.
    virtual const Value& value_for_key(const Key& key) const = 0;

protected:
    // Only sub-classes can instantiate.
    Dictionary() = default;
};
}

#endif  // MESSAGING_DICTIONARY_H_
