/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#ifndef MCLOUD_API_CLIENT_H_
#define MCLOUD_API_CLIENT_H_

#include <mcloud/api/visibility.h>
#include <mcloud/api/taskqueue.h>
#include <mcloud/api/diskinfo.h>
#include <mcloud/api/outlink.h>
#include <mcloud/api/cloudresource.h>
#include <mcloud/api/cloudcontent.h>
#include <mcloud/api/cloudfolder.h>
#include <mcloud/api/downloadtask.h>
#include <mcloud/api/uploadtask.h>
#include <mcloud/api/syncmanager.h>

#include <deque>
/*#include <future>*/
/*#include <thread>*/

namespace mcloud {
namespace api {

class ClientPriv;

/*!
    \class Client
    \brief Mcloud API provides developer to easily access login user's cloud contents stored on mcloud,
    such as musics, movie, docs, images, and so on.

    The API enable developers to
        <list type="bullet">
            <item>retrieve authorized user cloud content list.</item>
            <item>create, delete, move and update cloud resources.</item>
            <item>query content infomation.</item>
            <item>create an extranet link to share.</item>
            <item>upload a local content to mcloud.</item>
            <item>download a content from mcloud.</item>
        </list>

    Also sync up manager will be running on the background when client is constructed. It handles over
    local content upload and cloud item download. \sa syncmanager.

    \note Mcloud API covers cloud resourece access and content sync-up between local and cloud. It does not take
    responsibility of account authentication. So before using mcloud APIs, developers need to call \sa set_access_token to
    manually set a valid access_token or \sa refresh_token to fetch access token, 
    which is used to call mcloud APIs internally to pass through authentication.
 */

class MCLOUD_API_DLL_PUBLIC Client {

  public:
    typedef std::shared_ptr<Client> Ptr;

    typedef std::deque<CloudResource::Ptr>   ResourceList;

    typedef std::deque<Outlink::Ptr>    OutlinkList;

    typedef std::vector<std::string>    Stringlist;

    /*!
     * \brief Constructs a Mcloud Client object with \a request_timeout seconds.
     * Meanwhile client launches sync-up manager to deal with task download and upload by default.
     */
    Client(int request_timeout = 10);

    Client(const Client&) = delete;

    Client& operator=(const Client&) = delete;

    virtual ~Client();

    /*!
     * \brief Set \a access_token to authenticate all mcloud API calls with the bearer HTTP authorization scheme.
     * The access token can be fetched through ubuntu sso services.
     */
    void set_access_token(const std::string &access_token);

    /*!
     * \brief Retrieves and refreshes access token by using \a refresh_token.
     * The refresh token can be fetched through ubuntu sso services.
     * \throw std::runtime_error if error occurs.
     * \returns true if access token is refreshed, otherwise returns false.
     */
    bool refresh_token(const std::string &refresh_token);

    /*!
     * \brief Returns login user's clcoud storage info.
     * \throw std::runtime_error if error occurs.
     * @sa DiskInfo
     */
    DiskInfo disk_info();

    /*!
     * \brief Returns root folder id on mlcoud for third-party app to use, such as mcloud-lib.
     * \throw std::runtime_error if error occurs.
     * \note there're a few folders on a cmcc user's mcloud directory, but only one can be used for
     * content synchronization by third-party client. We regard it as the root folder for mcloud thrid-party app.
     * \sa cloud_content_list()
     */
    std::string cloud_root_folder_id();

    /*!
     * \brief Returns a cloud item list under one specific folder \a folder_id with a given type \a content_type.
     * if \a folder_id is empty, returns contents under root folder of cloud. \a content_type is
     * CloudContent::Type::All by default. The fetched content count is 200 at most per call.
     * \a start_index indicates the first index of the requested content list on mcloud.
     * \a count indicates the maximum number of content items per call.
     * \throw std::runtime_error if error occurs.
     * \sa CloudContent::Type cloud_root_folder_id()
     */
    ResourceList cloud_content_list(int start_index,
                                    int count,
                                    CloudContent::Type content_type = CloudContent::Type::All,
                                    const std::string &folder_id = std::string());

    /*!
     * \brief Returns a cloud content object if a content \a content_id exists on mcloud.
     * otherwise return nullptr.
     * \throw std::runtime_error if error occurs.
     * \sa delete_contents(), move_items()
     */
    CloudContent::Ptr   content_info(const std::string & content_id);

    /*!
     * \brief Returns a cloud folder object if a folder with \a folder_name is created under a given \a folder_id folder.
     * otherwise return nullptr.
     * \throw std::runtime_error if error occurs.
     * \sa delete_contents(), move_items()
     */
    CloudFolder::Ptr    create_folder(const std::string &folder_name,
                                      const std::string &folder_id);

    /*!
     * \brief Returns a cloud item list if there are available items that matches with \a name under a given \a folder_id folder.
     * \a property holds the property of cloud item, it can be content(default) or folder.
     * \throw std::runtime_error if error occurs.
     * \sa delete_contents(), move_items(), metadata()
     */

    [[gnu::deprecated("cmcc closes this api for security reason")]]
    Client::ResourceList    look_up(const std::string &name,
                                    const std::string &folder_id,
                                    CloudResource::Property property = CloudResource::Property::Content);

    /*!
     * \brief Creates folders extranet link object with a given \a folder_ids.
     * \throw std::runtime_error if error occurs.
     * \sa create_content_extranet_link()
     * \return a list of Outlink after extranet links are created.
     */
    OutlinkList create_folder_extranet_link(const Stringlist &folder_ids);

    /*!
     * \brief Creates a content_extranet link object with a given \a content_ids.
     * \throw std::runtime_error if error occurs.
     * \sa create_folder_extranet_link()
     * \return a list of Outlink after extranet links are created.
     */
    OutlinkList create_content_extranet_link(const Stringlist &content_ids);

    /*!
     * \brief Returns folder id list if folders \a folder_ids are copies to another folder with a given \a folder_id on mcloud
     * \throw std::runtime_error if error occurs.
     * \note the destination folder \a folder_id should not be the same as the folders' \a folder_ids folder ids.
     * \sa move_items(), update_folder(), copy_contents()
     */
    Client::Stringlist copy_folders(const Stringlist  &folder_ids,
                                    const std::string &_folder_id);

    /*!
     * \brief Returns content id list if contents \a content_ids are copies to another folder with a given \a folder_id on mcloud.
     * \throw std::runtime_error if error occurs.
     * \note the destination folder \a folder_id should not be the same as the contents' \a contents_ids folder id.
     * \sa move_items(), update_folder(), copy_folders()
     */
    Client::Stringlist copy_contents(const Stringlist  &contents_ids,
                                     const std::string &folder_id);

    /*!
     * \brief Returns true if \a folders with a given folder_ids and \a contents with a given \a content_ids
     * are successfully moved to new folder with a given \a folder_id.
     * \throw std::runtime_error if error occurs.
     * \sa delete_contents(), copy_folders(), copy_contents()
     */
    bool move_items(const Stringlist &folder_ids,
                    const Stringlist &content_ids,
                    const std::string &folder_id);

    /*!
     * \brief Returns true if folder \a folder_id are updated with \a new_folder_name, otherwise return false.
     * \throw std::runtime_error if error occurs.
     * \sa delete_contents(), copy_folders(),
     */
    bool update_folder(const std::string &folder_id,
                       const std::string &new_folder_name);

    /*!
     * \brief Returns true if contents \a content_ids are deleted, otherwise return false.
     * \throw std::runtime_error if error occurs.
     * \sa move_items(), content_info(), copy_contents()
     */
    bool delete_contents(const Stringlist &content_ids);

    /*!
     * \brief Returns true if a local file \a file_path exists in folder with \a folder_id on cloud, 
     * otherwise return false. Searches root folder if folder_id is empty();
     * \throw std::runtime_error if error occurs.
     */
    bool exist_on_cloud(const std::string &file_path, 
                        const std::string &folder_id = std::string());

    /*!
     * \brief Returns a sync-up manager running on the background.
     * sync-up manager scheduled tasks \sa start, pause, cancel for content upload and download.
     */
    SyncManager::Ptr syncmanager() const;

  private:
    std::shared_ptr<ClientPriv>  p_;

};

}
}

#endif // MCLOUD_API_CLIENT_H_
