# -*- coding: utf-8 -*-

# Author: Natalia Bidart <nataliabidart@gmail.com>
#
# Copyright 2011 Chicharreros
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Magicicada GTK UI."""

from __future__ import division

import logging
import time

from gettext import gettext as _

import cairo
import gtk
import pango

from magicicada import queue_content
from magicicada.gui.gtk.helpers import Buildable

ELLIPSIS = '...'
EXPAND_THRESHOLD = 15
FILE_ICON_NAME = 'text-x-generic'
FOLDER_ICON_NAME = 'folder'
HOME_ICON_NAME = 'user-home'
OPS_COMPLETED = 'Completed %s ago (%s)'
OPS_MARKUP = '<span foreground="#808080">%s</span>'
MAX_OP_LEN = 30
REMOTE_ICON_NAME = 'folder-remote'
TIME_UNITS = {0: _('second'), 1: _('minute'), 2: _('hour'), 3: _('day')}


logger = logging.getLogger('magicicada.gui.gtk.operations')


# Instance of 'A' has no 'y' member
# pylint: disable=E1101

# Unused argument, we need them for GTK callbacks
# pylint: disable=W0613


def pixbuf_from_icon_name(name):
    """Build a pixbuf for a given icon name."""
    icon = gtk.IconTheme()
    flags = gtk.ICON_LOOKUP_GENERIC_FALLBACK | gtk.ICON_LOOKUP_USE_BUILTIN
    pixbuf = icon.load_icon(name, 24, flags)
    return pixbuf


class NumberedFolder(gtk.gdk.Pixbuf):
    """A folder icon with a number count in it."""

    # NumberedFolder: Class has no __init__ method
    # pylint: disable=W0232

    @classmethod
    def new_from_amount(cls, amount):
        """Build a pixbuf with a number in it matching 'amount'."""
        pixbuf = pixbuf_from_icon_name('folder')

        cairo_format = cairo.FORMAT_RGB24
        if pixbuf.get_has_alpha():
            cairo_format = cairo.FORMAT_ARGB32

        width = pixbuf.get_width()
        height = pixbuf.get_height()
        surface = cairo.ImageSurface(cairo_format, width, height)
        context = gtk.gdk.CairoContext(cairo.Context(surface))
        context.set_source_pixbuf(pixbuf, 0, 0)
        context.paint()

        ll = context.create_layout()
        ##ll.set_width(width)
        ll.set_text(str(amount))
        ll.set_font_description(pango.FontDescription('Ubuntu Bold 8'))
        ##ll.set_alignment(pango.ALIGN_RIGHT)
        context.move_to(5, 5)
        context.set_source_rgb(0, 0, 0)
        context.update_layout(ll)
        context.show_layout(ll)

        data = surface.get_data()
        result = gtk.gdk.pixbuf_new_from_data(data, pixbuf.get_colorspace(),
                    pixbuf.get_has_alpha(), pixbuf.get_bits_per_sample(),
                    width, height, pixbuf.get_rowstride())

        return result


class Operations(Buildable, gtk.Alignment):
    """The list of operations over files/folders."""

    filename = 'operations.ui'

    def __init__(self, syncdaemon_instance):
        Buildable.__init__(self)
        gtk.Alignment.__init__(self, xscale=1, yscale=1)
        self.add(self.itself)
        self.ops_store = self.builder.get_object('ops_store')
        self._can_clear = False

        self.sd = syncdaemon_instance
        self.sd.on_node_ops_changed_callback = self.on_node_ops_changed

        self.clear_button.set_sensitive(False)
        self.show_all()

    def _process_operations(self, info):
        """Return the string to be shown as operations summary.

        The result may contain pango markup.

        """
        if not info.operations:
            return ''

        result = ', '.join([i[1] for i in info.operations
                            if not i[2][queue_content.DONE]])
        if not result:
            ops = ', '.join([i[1] for i in info.operations])
            if len(ops) > MAX_OP_LEN:
                ops = ops[:MAX_OP_LEN - len(ELLIPSIS)] + ELLIPSIS
            ago = int(time.time() - info.last_modified)
            unit = 0
            while ago >= 60:
                ago = ago // 60
                unit += 1
                if unit not in TIME_UNITS:
                    break

            ago = "%s %s%s" % (ago, TIME_UNITS[unit], 's' if ago > 1 else '')

            result = OPS_COMPLETED % (ago, ops)
            self._can_clear = True

        return OPS_MARKUP % result

    def _append_file_row(self, file_name, file_info, parent):
        """Append a new row to the store representing a file."""
        assert file_info.children == {}
        row = (file_name, self._process_operations(file_info),
               None, FILE_ICON_NAME, gtk.ICON_SIZE_SMALL_TOOLBAR)
        self.ops_store.append(parent, row)

    def _append_folder_row(self, folder_name, folder_info, parent):
        """Append a new row to the store representing a folder."""
        ## Won't use the NumberedFolder pixbuf until the TGB issue is resolved
        ##pixbuf = NumberedFolder.new_from_amount(children)
        row = (folder_name, self._process_operations(folder_info),
               None, FOLDER_ICON_NAME, gtk.ICON_SIZE_SMALL_TOOLBAR)
        parent = self.ops_store.append(parent, row)

        self._append_root_row(folder_info.children, parent)

        ## Expanding/collpasing is not working properly yet
        ##tree_path = self.ops_store.get_path(parent)
        ##children = len(folder_info.children)
        ##if children <= EXPAND_THRESHOLD:
        ##    self.ops_view.expand_to_path(tree_path)
        ##else:
        ##    self.ops_view.collapse_row(tree_path)

    def _append_root_row(self, root_info, parent):
        """Append a new row to the store representing a root share."""
        info = sorted(root_info.iteritems(),
                      key=lambda (name, data): (data.kind, name))
        for child_name, child_info in info:
            if child_info.kind == queue_content.KIND_DIR:
                self._append_folder_row(child_name, child_info, parent)
            else:
                self._append_file_row(child_name, child_info, parent)

    def on_node_ops_changed(self, items, *args, **kwargs):
        """Callback'ed when syncadaemon's node ops info changed."""
        if not items:
            items = []

        self.ops_view.set_model(None)
        self.ops_store.clear()
        self._can_clear = False

        for root_kind, root_info in items:
            # icon name should depend on root_kind
            icon_name = HOME_ICON_NAME \
                        if root_kind == queue_content.ROOT_HOME \
                        else REMOTE_ICON_NAME
            row = ('', '', None, icon_name, gtk.ICON_SIZE_LARGE_TOOLBAR)
            parent = self.ops_store.append(None, row)
            self._append_root_row(root_info, parent)

        self.ops_view.set_model(self.ops_store)
        #self.ops_view.expand_row('0', False)
        self.ops_view.expand_all()

        self.clear_button.set_sensitive(self._can_clear)

        self.show_all()

    def on_clear_button_clicked(self, button):
        """The clear button was clicked, remove all complete operations."""
        self.sd.queue_content.clear()
        self.load()

    def load(self):
        """Update UI based on SD current state."""
        self.on_node_ops_changed(self.sd.queue_content.node_ops)
