#!/bin/sh
set -e

# Pretend that a new development release has opened and this machine is on it.
# This bumps the version listed in /etc/lsb-release, and adds a new release to
# distro-info.
#
# Rationale: software such as Juju makes decisions based on the Ubuntu release
# it is running on. As a consequence, it may break in the archive when a new
# development release is opened even though nothing changed. Examples: LP:
# #1314686, LP: #1321025. Testing a "future" release will hopefully make sure
# that Juju will do the right thing for a future as-yet-unknown release that is
# identical to this one (such as when a new development release opens).

# Blame Robie Basak <robie.basak@canonical.com> for this particularly twisted
# hack. Although in a similarly twisted way, I'm quite proud of it :)

info_file=/usr/share/distro-info/ubuntu.csv
lsb_file=/etc/lsb-release
os_release_file=/etc/os-release

# foo -> goo
# Trusty -> Urusty
# Utopic -> Vtopic
bump_word() {
	read r
	l=${r#?}
	f=${r%$l}
	n=`echo $f|tr a-y,z,A-Y,Z b-z,a,B-Z,A`
	echo "$n$l"
}

# 12.04 -> 13.04
bump_version() {
	read v
	maj=${v%.*}
	min=${v#*.}
	n=`expr 1 + $maj`
	echo "${n}.${min}"
}

# foo bar -> goo car
# Trusty Tahr -> Urusty Uahr
# Utopic Unicorn -> Vtopic Vnicorn
bump_words() {
	read words
	bumped=`for w in $words; do echo "$w"|bump_word;done`
	printf "%s" "$bumped"|tr "\n" ' '
	echo
}

last_line=`tail -1 "$info_file"`
version=`echo "$last_line"|cut -d, -f1|bump_version`
version_=`echo "$version"|awk '{print $1}'`
codename=`echo "$last_line"|cut -d, -f2|bump_words`
series=`echo "$last_line"|cut -d, -f3|bump_word`

created=`date -d '-2 days' +%Y-%M-%d`
release=`date -d '-1 days' +%Y-%M-%d`
eol=`date -d '+1 days' +%Y-%M-%d`

id=`lsb_release -si`

[ -f "${info_file}.orig" ] || cp -a "${info_file}" "${info_file}.orig"
echo "${version},${codename},${series},${created},${release},${eol},${eol}" >> /usr/share/distro-info/ubuntu.csv

[ -f "${lsb_file}.orig" ] || cp -a "${lsb_file}" "${lsb_file}.orig"
cat > "$lsb_file" <<EOT
DISTRIB_ID=$id
DISTRIB_RELEASE=$version_
DISTRIB_CODENAME=$series
DISTRIB_DESCRIPTION="$id $version"
EOT

[ -f "${os_release_file}.orig" ] || cp -a "${os_release_file}" "${os_release_file}.orig"
cat > "$os_release_file" <<EOT
NAME="Ubuntu"
VERSION="$version ($codename)"
ID=ubuntu
ID_LIKE=debian
PRETTY_NAME="Ubuntu $codename"
VERSION_ID="$version"
HOME_URL="http://www.ubuntu.com/"
SUPPORT_URL="http://help.ubuntu.com/"
BUG_REPORT_URL="http://bugs.launchpad.net/ubuntu/"
EOT
