# -*- coding: utf-8 -*-
import sys
import os
import string
import shapelib as shp
import math
import struct, datetime, decimal, itertools


'''
    Returns an iterator over records in a Xbase DBF file.

    The first row returned contains the field names.
    The second row contains field specs: (type, size, decimal places).
    Subsequent rows contain the data records.
    If a record is marked as deleted, it is skipped.

    File should be opened for binary reads.

'''
# See DBF format spec at:
#     http://www.pgts.com.au/download/public/xbase.htm#DBF_STRUCT
def dbfreader(f):

    numrec, lenheader = struct.unpack('<xxxxLH22x', f.read(32))    
    numfields = (lenheader - 33) // 32

    fields = []
    for fieldno in xrange(numfields):
        name, typ, size, deci = struct.unpack('<11sc4xBB14x', f.read(32))
        name = name.replace('\0', '')       # eliminate NULs from string   
        fields.append((name, typ, size, deci))
    yield [field[0] for field in fields]
    yield [tuple(field[1:]) for field in fields]

    terminator = f.read(1)
    assert terminator == '\r'

    fields.insert(0, ('DeletionFlag', 'C', 1, 0))
    fmt = ''.join(['%ds' % fieldinfo[2] for fieldinfo in fields])
    fmtsiz = struct.calcsize(fmt)
    for i in xrange(numrec):
        record = struct.unpack(fmt, f.read(fmtsiz))
        if record[0] != ' ':
            continue                        # deleted record
        result = []
        for (name, typ, size, deci), value in itertools.izip(fields, record):
            if name == 'DeletionFlag':
                continue
            if typ == "N":
                value = value.replace('\0', '').lstrip()
                if value == '':
                    value = 0
                elif deci:
                    pass # value = decimal.Decimal(value)
                else:
                    pass #value = int(value)
            elif typ == 'D':
                y, m, d = int(value[:4]), int(value[4:6]), int(value[6:8])
                value = datetime.date(y, m, d)
            elif typ == 'L':
                value = (value in 'YyTt' and 'T') or (value in 'NnFf' and 'F') or '?'
            elif typ == 'F':
                value = float(value)
            result.append(value)
        yield result

'''
    Ramen, Douglas, Peuker algorithm
'''
def simplify_points (pts, tolerance):
    anchor  = 0
    floater = len(pts) - 1
    stack   = []
    keep    = set()

    stack.append((anchor, floater))
    while stack:
        anchor, floater = stack.pop()

        # initialize line segment
        if pts[floater] != pts[anchor]:
            anchorX = float(pts[floater][0] - pts[anchor][0])
            anchorY = float(pts[floater][1] - pts[anchor][1])
            seg_len = math.sqrt(anchorX ** 2 + anchorY ** 2)
            # get the unit vector
            anchorX /= seg_len
            anchorY /= seg_len
        else:
            anchorX = anchorY = seg_len = 0.0

        # inner loop:
        max_dist = 0.0
        farthest = anchor + 1
        for i in range(anchor + 1, floater):
            dist_to_seg = 0.0
            # compare to anchor
            vecX = float(pts[i][0] - pts[anchor][0])
            vecY = float(pts[i][1] - pts[anchor][1])
            seg_len = math.sqrt( vecX ** 2 + vecY ** 2 )
            # dot product:
            proj = vecX * anchorX + vecY * anchorY
            if proj < 0.0:
                dist_to_seg = seg_len
            else:
                # compare to floater
                vecX = float(pts[i][0] - pts[floater][0])
                vecY = float(pts[i][1] - pts[floater][1])
                seg_len = math.sqrt( vecX ** 2 + vecY ** 2 )
                # dot product:
                proj = vecX * (-anchorX) + vecY * (-anchorY)
                if proj < 0.0:
                    dist_to_seg = seg_len
                else:  # calculate perpendicular distance to line (pythagorean theorem):
                    dist_to_seg = math.sqrt(abs(seg_len ** 2 - proj ** 2))
                if max_dist < dist_to_seg:
                    max_dist = dist_to_seg
                    farthest = i

        if max_dist <= tolerance: # use line segment
            keep.add(anchor)
            keep.add(floater)
        else:
            stack.append((anchor, farthest))
            stack.append((farthest, floater))

    keep = list(keep)
    keep.sort()
    return [pts[i] for i in keep]


if __name__ == '__main__':
    if len(sys.argv)<2:
        sys.stderr.write("call: python shp2jsx.py filename [factor] \n")
        sys.stderr.write("\t file name has to have the ending dbf.  \n")
        sys.exit(0) 
        
    filename = sys.argv[1]
    if not os.path.exists(filename):
        sys.stderr.write("file '%s' not found\n" % filename)
        sys.exit(0) 
    f = open(filename, "r")

        fac = 1.0                   #100000.0
    if len(sys.argv)>2:
        fac = float(sys.argv[2])
    
    f = shp.open(filename)          #  get handle to the the shape file
    nParts = f.info()[0]

    '''
        extract bounding box
    '''
    minx = f.info()[2][0]/fac
    miny = f.info()[2][1]/fac
    maxx = f.info()[3][0]/fac
    maxy = f.info()[3][1]/fac

    print "bbox = [%0.2f,%0.2f,%0.2f,%0.2f];" %(minx*0.99,maxy*1.01,maxx*1.01,miny*0.99)
    
    '''
        Read the paths information
    ''' 
    print "paths = [];";
    for n in xrange(nParts):
        id = f.read_object(n).id
        for parts in f.read_object(n).vertices():
            parts = simplify_points(parts, 1000.0)
            print "paths.push(["
            print "\t[",string.join(["%0.2f"%(p[0]/fac) for p in parts],','),'],'
            print "\t[",string.join(["%0.2f"%(p[1]/fac) for p in parts],','),'],'
            print "\t", id, "]);"

    '''
        Extract the text info
    '''
    print "info = []";
    f = open(filename, 'rb')
    db = list(dbfreader(f))
    f.close()

    i = 0
    for record in db:
        if i>1:
            print "info.push(", record, ");" 
        i += 1
