# jsb/persist.py
#
#

"""
    allow data to be written to disk or BigTable in JSON format. creating 
    the persisted object restores data. 

"""

## jsb imports

from jsb.utils.trace import whichmodule, calledfrom, callstack, where
from jsb.utils.lazydict import LazyDict
from jsb.utils.exception import handle_exception
from jsb.utils.name import stripname
from jsb.utils.locking import lockdec
from jsb.utils.timeutils import elapsedstring
from jsb.lib.callbacks import callbacks
from jsb.lib.errors import MemcachedCounterError

from datadir import getdatadir

## simplejson imports

from jsb.imports import getjson
json = getjson()

## basic imports

import thread
import logging
import os
import os.path
import types
import copy
import sys
import time

## defines

cpy = copy.deepcopy


## global list to keeptrack of what persist objects need to be saved

needsaving = []

def cleanup(bot=None, event=None):
    global needsaving
    r = []
    for p in needsaving:
        try: p.dosave() ; r.append(p) ; logging.warn("saved on retry - %s" % p.fn)
        except (OSError, IOError), ex: logging.error("failed to save %s - %s" % (p, str(ex)))
    for p in r:
        try: needsaving.remove(p)
        except ValueError: pass
    return needsaving

## try google first

try:
    import waveapi
    from google.appengine.ext import db
    import google.appengine.api.memcache as mc
    from google.appengine.api.datastore_errors import Timeout
    from cache import get, set, delete
    logging.debug("using BigTable based Persist")

    ## JSONindb class

    class JSONindb(db.Model):
        """ model to store json files in. """
        modtime = db.DateTimeProperty(auto_now=True, indexed=False)
        createtime = db.DateTimeProperty(auto_now_add=True, indexed=False)
        filename = db.StringProperty()
        content = db.TextProperty(indexed=False)

    ## Persist class

    class Persist(object):

        """ persist data attribute to database backed JSON file. """ 

        def __init__(self, filename, default={}, type="cache"):
            self.cachtype = None
            self.plugname = calledfrom(sys._getframe())
            if 'lib' in self.plugname: self.plugname = calledfrom(sys._getframe(1))
            try: del self.fn
            except: pass 
            self.fn = unicode(filename.strip()) # filename to save to
            self.logname = os.sep.join(self.fn.split(os.sep)[-1:])
            self.countername = self.fn + "_" + "counter"
            self.mcounter = mc.get(self.countername) or mc.set(self.countername, "1")
            try: self.mcounter = int(self.mcounter)
            except ValueError: logging.warn("can't parse %s mcounter, setting to zero: %s" % (self.fn, self.mcounter)) ; self.mcounter = 0
            self.data = None
            self.type = type
            self.counter = self.mcounter
            self.key = None
            self.obj = None
            self.size = 0
            self.jsontxt = None
            self.init(default)

        def init(self, default={}, filename=None):
            cachetype = ""
            if self.checkmc(): self.jsontxt = self.updatemc() ; self.cachetype = "cache"
            else:
                self.data = get(self.fn)
                self.cachetype = "mem"
                if self.data != None:
                    logging.debug("source %s - loaded %s (%s)" % (self.cachetype, self.fn, str(self.data)))
                    if type(self.data) == types.DictType: self.data = LazyDict(self.data)
                    return self.data
            if self.jsontxt is None:
                self.cachetype = "db"
                logging.debug("%s - loading from db" % self.fn) 
                try:
                    try: self.obj = JSONindb.get_by_key_name(self.fn)
                    except Timeout: self.obj = JSONindb.get_by_key_name(self.fn)
                except Exception, ex:
                    # bw compat sucks
                    try: self.obj = JSONindb.get_by_key_name(self.fn)
                    except Exception, ex:
                        handle_exception()
                        self.obj = None
                if self.obj == None:
                    logging.debug("%s - no entry found" % self.fn)
                    self.jsontext = json.dumps(default) ; self.cachetype = "default"
                else:
                    self.jsontxt = self.obj.content; self.cachetype = "db"
                if self.jsontxt:
                    mc.set(self.fn, self.jsontxt)
                    incr = mc.incr(self.countername)
                    if incr:
                        try: self.mcounter = self.counter = int(incr)
                        except ValueError: logging.error("can't make counter out of %s" % incr) 
                    else: self.mcounter = 1
            logging.info("memcached counters for %s: %s" % (self.fn, self.mcounter))
            if self.jsontxt == None: self.jsontxt = json.dumps(default) 
            logging.info('%s - jsontxt is %s' % (self.fn, self.jsontxt))
            self.data = json.loads(self.jsontxt)
            if not self.data: self.data = default
            self.size = len(self.jsontxt)
            if type(self.data) == types.DictType: self.data = LazyDict(self.data)
            set(self.fn, self.data)
            logging.debug("source %s - loaded %s (%s)" % (self.cachetype, self.fn, len(self.jsontxt)))

        def sync(self):
            logging.info("syncing %s" % self.fn)
            tmp = cpy(self.data)
            data = json.dumps(tmp)
            mc.set(self.fn, data)
            if type(self.data) == types.DictType:
                self.data = LazyDict(self.data)
            set(self.fn, self.data)
            return data

        def updatemc(self):
            tmp = mc.get(self.fn)
            if tmp != None:
                try:
                    t = json.loads(tmp)
                    if self.data: t.update(self.data)
                    self.data = LazyDict(t)
                    logging.warn("updated %s" % self.fn)
                except AttributeError, ex: logging.warn(str(ex))
                return self.data

        def checkmc(self):
            try:
                self.mcounter = int(mc.get(self.countername)) or 0
            except: self.mcounter = 0
            logging.debug("mcounter for %s is %s" % (self.fn, self.mcounter))
            if (self.mcounter - self.counter) < 0: self.mcounter = self.counter = 0 ; return True
            elif (self.mcounter - self.counter) > 0: return True
            return False
  
        def save(self, filename=None):
            """ save json data to database. """
            if self.checkmc(): self.updatemc()
            fn = filename or self.fn
            bla = json.dumps(self.data)
            if filename or self.obj == None:
                self.obj = JSONindb(key_name=fn)
                self.obj.content = bla
            else: self.obj.content = bla
            self.obj.filename = fn
            from google.appengine.ext import db
            key = db.run_in_transaction(self.obj.put)
            logging.debug("transaction returned %s" % key)
            mc.set(fn, bla)
            if type(self.data) == types.DictType: self.data = LazyDict(self.data)
            set(fn, self.data)
            incr = mc.incr(self.countername)
            if incr:
                try: self.mcounter = self.counter = int(incr)
                except ValueError: logging.error("can't make counter out of %s" % incr) 
            else: self.mcounter = 1
            self.counter = self.mcounter
            logging.info("memcached counters for %s: %s" % (fn, self.mcounter))
            logging.warn('saved %s (%s) - %s' % (fn, len(bla), where()))

        def upgrade(self, filename):
            self.init(self.data, filename=filename)


except ImportError:

    ## file based persist

    logging.debug("using file based Persist")

    ## defines

    persistlock = thread.allocate_lock()
    persistlocked = lockdec(persistlock)

    ## imports for shell bots

    if True:
        got = False
        from jsb.memcached import getmc
        mc = getmc()
        if mc:
            status = mc.get_stats()
            if status:
                logging.warn("memcached uptime is %s" % elapsedstring(status[0][1]['uptime']))
                got = True
        if got == False:
            logging.info("no memcached found - using own cache")
        from cache import get, set, delete

    import fcntl

    ## classes

    class Persist(object):

        """ persist data attribute to JSON file. """
        
        def __init__(self, filename, default=None, init=True, postfix=None):
            """ Persist constructor """
            if postfix: self.fn = str(filename.strip()) + str("-%s" % postfix)
            else: self.fn = str(filename.strip())
            self.lock = thread.allocate_lock() # lock used when saving)
            self.data = LazyDict(default=default) # attribute to hold the data
            try:
                res = []
                target = getdatadir().split(os.sep)[-1]
                for i in self.fn.split(os.sep)[::-1]:
                    if target in i: break
                    res.append(i)
                self.logname = os.sep.join(res[::-1])
                if not self.logname: self.logname = self.fn
            except: handle_exception() ; self.logname = self.fn
            self.countername = self.fn + "_" + "counter"
            if got:
                count = mc.get(self.countername)
                try:
                    self.mcounter = self.counter = int(count)
                except (ValueError, TypeError):
                    self.mcounter = self.counter = mc.set(self.countername, "1") or 0
            else:
                self.mcounter = self.counter = 0
            self.ssize = 0
            self.jsontxt = ""
            self.dontsave = False
            if init:
                self.init(default)
                if default == None: default = LazyDict()

        def size(self):
            return "%s (%s)" % (len(self.data), len(self.jsontxt))

        def init(self, default={}, filename=None):
            """ initialize the data. """
            gotcache = False
            cachetype = "cache"
            try:
                logging.debug("using name %s" % self.fn)
                a = get(self.fn)
                if a: self.data = a.data
                else: self.data = None
                if self.data != None:
                    logging.debug("got data from local cache")
                    return self
                if got: self.jsontxt = mc.get(self.fn) ; cachetype = "cache"
                if not self.jsontxt:
                   datafile = open(self.fn, 'r')
                   self.jsontxt = datafile.read()
                   datafile.close()
                   self.ssize = len(self.jsontxt)
                   cachetype = "file"
                   if got: mc.set(self.fn, self.jsontxt)
            except IOError, ex:
                if not 'No such file' in str(ex):
                    logging.error('failed to read %s: %s' % (self.fn, str(ex)))
                    raise
                else:
                    logging.debug("%s doesn't exist yet" % self.fn)
                    self.jsontxt = json.dumps(default)
            try:
                if self.jsontxt:
                    logging.debug(u"loading: %s" % type(self.jsontxt))
                    try: self.data = json.loads(str(self.jsontxt))
                    except Exception, ex: logging.error("couldn't parse %s" % self.jsontxt) ; self.data = None ; self.dontsave = True
                if not self.data: self.data = LazyDict()
                elif type(self.data) == types.DictType:
                    logging.debug("converting dict to LazyDict")
                    d = LazyDict()
                    d.update(self.data)
                    self.data = d
                set(self.fn, self)
                logging.info("loaded %s - %s" % (self.logname, cachetype))
            except Exception, ex:
                logging.error('ERROR: %s' % self.fn)
                raise

        def upgrade(self, filename):
            self.init(self.data, filename=filename)
            self.save(filename)

        def get(self):
            logging.debug("getting %s from local cache" % self.fn)
            a = get(self.fn)
            logging.debug("got %s from local cache" % type(a))
            return a

        def sync(self):
            logging.info("syncing %s" % self.fn)
            if got: mc.set(self.fn, json.dumps(self.data))
            set(self.fn, self)
            return self

        def save(self):
            global needsaving
            try: self.dosave()
            except (IOError, OSError):
                self.sync()
                if self not in needsaving: needsaving.append(self)
        
        @persistlocked
        def dosave(self):
            """ persist data attribute. """
            try:
                if self.dontsave: logging.error("dontsave is set on  %s - not saving" % self.fn) ; return
                fn = self.fn
                #print help(mc.incr)
                if got: self.mcounter = int(mc.incr(self.countername))
                if got and (self.mcounter - self.counter) > 1:
                    tmp = json.loads(mc.get(fn))
                    if tmp:
                        try: tmp.update(self.data) ; self.data = LazyDict(tmp) ; logging.warn("updated %s" % fn)
                        except AttributeError: pass
                    self.counter = self.mcounter
                d = []
                if fn.startswith(os.sep): d = [os.sep,]
                for p in fn.split(os.sep)[:-1]:
                    if not p: continue
                    d.append(p)
                    pp = os.sep.join(d)
                    if not os.path.isdir(pp):
                        logging.info("creating %s dir" % pp)
                        os.mkdir(pp)
                tmp = fn + '.tmp' # tmp file to save to
                datafile = open(tmp, 'w')
                fcntl.flock(datafile, fcntl.LOCK_EX | fcntl.LOCK_NB)
                json.dump(self.data, datafile, indent=True)
                fcntl.flock(datafile, fcntl.LOCK_UN)
                datafile.close()
                try: os.rename(tmp, fn)
                except (IOError, OSError):
                    os.remove(fn)
                    os.rename(tmp, fn)
                jsontxt = json.dumps(self.data)
                logging.debug("setting cache %s - %s" % (fn, jsontxt))
                self.jsontxt = jsontxt
                set(fn, self)
                if got: mc.set(fn, jsontxt)
                logging.warn('%s saved' % self.logname)
            except IOError, ex: logging.error("not saving %s: %s" % (self.fn, str(ex))) ; raise
            except: raise
            finally: pass

class PlugPersist(Persist):

    """ persist plug related data. data is stored in jsondata/plugs/{plugname}/{filename}. """

    def __init__(self, filename, default={}, *args, **kwargs):
        plugname = calledfrom(sys._getframe())
        Persist.__init__(self, getdatadir() + os.sep + 'plugs' + os.sep + stripname(plugname) + os.sep + stripname(filename), default=default, *args, **kwargs)

class GlobalPersist(Persist):

    """ persist plug related data. data is stored in jsondata/plugs/{plugname}/{filename}. """

    def __init__(self, filename, default={}, *args, **kwargs):
        if not filename: raise Exception("filename not set in GlobalPersist")
        logging.warn("filename is %s" % filename)
        Persist.__init__(self, getdatadir() + os.sep + 'globals' + os.sep + stripname(filename), default=default, *args, **kwargs)

callbacks.add("TICK60", cleanup)
