/*
 * This code was initially written by Rudolf Fruehwirth (Institut fuer
 * Hochenergiephysik Oesterreichische Akademie der Wissenschaften,
 * Nikolsdorfer Gasse 18, 1050 Wien), and later adapted to GMRFLib by
 * H.Rue, then adapted to JAGS by Martyn Plummer.
 *
 * This routine approximates the negative log-gamma distribution, with density
 * 
 * exp(-n*x - exp(-x)) / Gamma(n)
 * 
 * by a mixture of normals
 * 
 * \sum_{i=1}^N weights[i] * N(x; means[i], variances[i])
 * 
 * with N components.
 */
 
#include <config.h>
#include "LGMix.h"
#include <JRmath.h>
#include <rng/RNG.h>
#include <module/ModuleError.h>

#include <cmath>
#include <vector>

using std::sqrt;
using std::vector;

/*
 * Exact mixture parameters are stored for integer n in the range 1 ... 19.
 *
 * For n = 1 ... 4, 10 mixture components are required and the
 * parameters are stored in the 4 x 10 arrays P10 (weights), M10
 * (means) and V10 (variances)
 *
 * For n = 5 ... 19, 9 components are required and the parameters are
 * stored in the 15 x 9 arrays P9 (weights), M9 (means) and V9
 * (variances)
 */

static const double P10[4][10] = {
    {0.00396984425, 0.0396244597, 0.16776747, 0.147036501, 0.125306271,
     0.1014852, 0.103758531, 0.115972617, 0.107065659, 0.0880134482},
    {0.00396860865, 0.039627049, 0.16777003, 0.147037759, 0.125304523,
     0.101481714, 0.103759705, 0.115973128, 0.107065554, 0.0880119305},
    {0.00396904734, 0.0396280642, 0.167770514, 0.14703607, 0.12530043,
     0.10148242, 0.103759287, 0.115974323, 0.107066971, 0.0880128738},
    {0.00396883344, 0.039627504, 0.167771274, 0.147036659, 0.125301189,
     0.101481755, 0.103760036, 0.115974339, 0.107065718, 0.0880126919}
};
static const double M10[4][10] = {
    {3.55887454, 2.11415904, 0.968124631, 0.51537638, 0.145465449,
     -0.145346445, -0.416660312, -0.689002855, -0.974965634, -1.27310004},
    {2.78807754, 1.84979328, 0.94844169, 0.577673108, 0.223449219,
     -0.0831666379, -0.387174155, -0.69613969, -1.01843553, -1.34112844},
    {2.43454312, 1.7315327, 0.942157786, 0.60208557, 0.251664821,
     -0.0644746918, -0.379817508, -0.696781518, -1.0293035, -1.35705784},
    {2.30484474, 1.71231656, 0.952907078, 0.601128034, 0.252368847,
     -0.059032783, -0.375605704, -0.699071542, -1.03734211, -1.3609072}
};
static const double V10[4][10] = {
    {2.62603032, 1.21263644, 0.66586521, 0.256650604, 0.120071142,
     0.0649909219, 0.0473513798, 0.046639443, 0.0576541602, 0.0888536903},
    {2.39619753, 1.16995764, 0.688870128, 0.307084756, 0.155644328,
     0.0899360571, 0.0707828448, 0.0751755614, 0.0990773728, 0.15471843},
    {2.16215586, 1.11062998, 0.682294453, 0.324750601, 0.173204837,
     0.108063698, 0.0917073596, 0.100257256, 0.131371692, 0.200024832},
    {1.92939547, 1.00671896, 0.638983371, 0.322852776, 0.18445103,
     0.122217472, 0.106400052, 0.116936918, 0.154113316, 0.233525098}
};

static const double P9[15][9] = {
    {0.0435820277, 0.167794347, 0.147040722, 0.125310654, 0.10147112,
     0.10376347, 0.115973878, 0.107056197, 0.0880075845},
    {0.0435817033, 0.167795985, 0.1470426, 0.125311016, 0.101470666,
     0.103763084, 0.115972864, 0.107055437, 0.0880066471},
    {0.0435798639, 0.167797087, 0.147042073, 0.125313291, 0.101470979,
     0.103761847, 0.115973234, 0.107054351, 0.0880072753},
    {0.043578895, 0.167797426, 0.147041988, 0.125313875, 0.101470922,
     0.103761581, 0.115973137, 0.107054001, 0.0880081751},
    {0.0435786725, 0.167797743, 0.1470428, 0.125313553, 0.101470946,
     0.103761391, 0.115973188, 0.10705364, 0.0880080663},
    {0.0435779307, 0.167797788, 0.147042734, 0.125314068, 0.101471449,
     0.10376142, 0.115973187, 0.107053473, 0.0880079505},
    {0.043576761, 0.167801375, 0.147042624, 0.125314075, 0.101470546,
     0.103761069, 0.115973226, 0.107051966, 0.0880083593},
    {0.0435771819, 0.167801103, 0.147042441, 0.125313864, 0.101470305,
     0.103761519, 0.11597319, 0.107052417, 0.0880079809},
    {0.0435778469, 0.167800486, 0.147041951, 0.125313914, 0.101470076,
     0.103761707, 0.115973611, 0.107052756, 0.0880076518},
    {0.0435786417, 0.16779926, 0.147042119, 0.125313391, 0.101470554,
     0.103762378, 0.115973792, 0.107052537, 0.0880073289},
    {0.043581505, 0.167797871, 0.147043608, 0.125312521, 0.101469081,
     0.103762173, 0.115973414, 0.107054363, 0.0880054639},
    {0.0435811435, 0.167798952, 0.147043687, 0.125312616, 0.101468918,
     0.103762052, 0.115973417, 0.107053968, 0.0880052462},
    {0.0435812603, 0.167798873, 0.147044518, 0.125312321, 0.101468879,
     0.103761729, 0.115972692, 0.107054049, 0.0880056789},
    {0.0435808733, 0.167799002, 0.147044529, 0.125312675, 0.101468951,
     0.103761472, 0.115972643, 0.107053883, 0.0880059719},
    {0.0435807283, 0.167799231, 0.14704464, 0.12531292, 0.101468814,
     0.103761275, 0.115972628, 0.107053662, 0.088006103}
};
static const double M9[15][9] = {
    {1.31113348, 0.963928895, 0.659198795, 0.240742429, -0.108844644,
     -0.252087404, -0.6546691, -1.04146524, -1.37874376},
    {1.25919247, 0.957217299, 0.66710982, 0.251658342, -0.125234491,
     -0.240137829, -0.64912733, -1.03921002, -1.37439461},
    {1.21602216, 0.94778507, 0.671484869, 0.265435387, -0.104709908,
     -0.24708343, -0.653441223, -1.04076324, -1.36988994},
    {1.18027937, 0.939725546, 0.67760436, 0.293497817, -0.110879079,
     -0.257696481, -0.655613756, -1.0406543, -1.36465528},
    {1.14996911, 0.934206664, 0.686267712, 0.311595579, -0.112948479,
     -0.274222612, -0.653808807, -1.04092104, -1.35962481},
    {1.12841748, 0.932206841, 0.69102714, 0.319038554, -0.109581301,
     -0.302963892, -0.641448217, -1.03858769, -1.35274157},
    {1.10451126, 0.925180162, 0.689947194, 0.309587296, -0.123979787,
     -0.239246368, -0.658582798, -1.03932069, -1.347407},
    {1.08624068, 0.918081034, 0.697616213, 0.330655882, -0.106424319,
     -0.290644969, -0.644517493, -1.04099153, -1.34370607},
    {1.0671125, 0.915784215, 0.70024231, 0.330800476, -0.125598534,
     -0.244656951, -0.661886313, -1.04447342, -1.33948264},
    {1.05721516, 0.918099637, 0.698999193, 0.325014717, -0.153165358,
     -0.225909041, -0.659788653, -1.03711782, -1.33064663},
    {1.02150943, 0.896206397, 0.702224917, 0.344137939, -0.119895501,
     -0.256590721, -0.641185291, -1.03810889, -1.32943558},
    {1.02508782, 0.902555642, 0.699256309, 0.336391119, -0.121902141,
     -0.242730179, -0.6538063, -1.0385784, -1.32415888},
    {0.997274184, 0.88197491, 0.696155279, 0.3460138, -0.128981232,
     -0.227346713, -0.630322077, -1.03647508, -1.32316505},
    {0.995086849, 0.891409674, 0.70171109, 0.341992158, -0.127906113,
     -0.245952673, -0.638792902, -1.03392281, -1.31486719},
    {0.997741814, 0.892112396, 0.698155553, 0.337731787, -0.122630195,
     -0.240878604, -0.651951415, -1.02898878, -1.3062535}
};

static const double V9[15][9] = {
    {1.5732832, 0.745075965, 0.340530976, 0.206325108, 0.206977107,
     0.133034557, 0.123981078, 0.155417698, 0.247661591},
    {1.52550277, 0.745216293, 0.347702459, 0.213195645, 0.220928839,
     0.147502243, 0.139478204, 0.17271313, 0.269719569},
    {1.48970429, 0.74910777, 0.35810967, 0.221696291, 0.216470192,
     0.155837875, 0.148481868, 0.185394632, 0.28822907},
    {1.46105103, 0.752441091, 0.365198621, 0.220104509, 0.199190433,
     0.167708126, 0.15761138, 0.197076001, 0.304425302},
    {1.43764551, 0.754190306, 0.367534375, 0.215776065, 0.185257157,
     0.180142183, 0.165402413, 0.206954388, 0.318591695},
    {1.41468216, 0.75198881, 0.368357589, 0.215271168, 0.178178434,
     0.198636491, 0.176790288, 0.218155881, 0.332156859},
    {1.39851898, 0.755429842, 0.377058085, 0.229287048, 0.214645547,
     0.18489307, 0.178139004, 0.226237823, 0.343708183},
    {1.38111403, 0.759024378, 0.379809227, 0.222659694, 0.185443843,
     0.206181273, 0.184773494, 0.231840962, 0.353714302},
    {1.36922993, 0.759197249, 0.381687395, 0.225704876, 0.199623554,
     0.195711194, 0.18270427, 0.236837387, 0.363050264},
    {1.35398708, 0.753650144, 0.381381699, 0.231057971, 0.208319112,
     0.210190241, 0.194957855, 0.249236388, 0.373774124},
    {1.35652837, 0.774748407, 0.400413698, 0.238592235, 0.199467639,
     0.230239828, 0.19924794, 0.251600772, 0.380054821},
    {1.33546695, 0.763749521, 0.396745563, 0.241905327, 0.212176877,
     0.218950701, 0.201882762, 0.257807637, 0.388524892},
    {1.33575722, 0.781739895, 0.417799104, 0.256728889, 0.211230256,
     0.254750255, 0.208700024, 0.26087813, 0.393822372},
    {1.3227643, 0.771070524, 0.406631212, 0.249617029, 0.210958958,
     0.249496089, 0.214362668, 0.270024593, 0.402615529},
    {1.30630549, 0.765952536, 0.407914566, 0.255018833, 0.226289944,
     0.236887588, 0.221124118, 0.280039124, 0.411219814}
};


/*
 * For n >= 20, approximate mixture parameters are calculated using a
 * rational approximation (see function rational_approx below). The
 * coefficients of the rational function are stored in an ncomp x 4
 * array, where ncomp is the number of components in the approximation.
 * 
 * Separate coefficents are used for n in the ranges:
 * (20...49) - 4 components
 * (50...439) - 3 components
 * (440...1599) - 2 components
 * (1600...9999) - 2 components
 * (1000 ... 30000) - 2 components
 *
 * For n > 30000, only a single component is required. In other words
 * the log-gamma distribution is approximated well by a single normal
 * distribution.
 *
 * Note that the coefficients for (20...49) are given to 12 decimal
 * places, whereas the others are given to 15 decimal places. This is
 * the way they were given in the GMRF library.
 */

/*
 * n from 20 to 49 
 */
static const double Coef_p3[4][4] = {
    {-5.644536495326e-09, -1.266992312621e-06, 
     0.000000000000e+00, 4.730022618640e+00},
    {7.299190941772e-09, 1.387196986613e-06, 
     0.000000000000e+00, 3.672627139064e+00},
    {-1.788056445701e-08, -2.391966642312e-06, 
     0.000000000000e+00, 4.871566292199e+00},
    {9.259794020097e-09, 1.224613603301e-06, 
     0.000000000000e+00, 3.215154075256e+00}
};
static const double Coef_m3[4][4] = {
    {4.552797222246e-05, 4.638009105861e-02, 
     9.627160143020e-02, 1.143772956136e+00},
    {2.284729919322e-05, -1.095058888700e-02, 
     -1.690501643196e-02, 1.944583776810e+00},
    {-3.900177124794e-05, 4.731686443506e-02, 
     -5.610095109269e-01, -6.033854619021e+00},
    {-2.486737015928e-05, 2.978371498898e-02, 
     -4.643825308040e-02, -1.105498133467e+00}
};
static const double Coef_v3[4][4] = {
    {-2.191015160635e-05, 9.939739739229e-02, 
     9.208564449364e-02, 7.148740127686e-01},
    {7.060864706965e-05, 1.143203813438e-01, 
     1.548617268518e-01, 2.428636911969e+00},
    {1.823003483481e-04, 1.675101633325e-01, 
     2.735383307281e-01, 4.861423133312e+00},
    {1.613752763707e-04, 1.943336591437e-01, 
     2.797653349940e-01, 3.840341872065e+00}
};

/*
 * n from 50 to 439 
 */
static const double Coef_p4[3][4] = {
    {-5.639545796991280e-10, 4.698743002874532e-07, 
     0.000000000000000e+00, 4.730482920811330e+00},
    {2.651836392450035e-10, -1.280380156002802e-07, 
     0.000000000000000e+00, 2.093982718501769e+00},
    {-2.384482520627535e-11, -1.227680572544847e-07, 
     0.000000000000000e+00, 3.214956149674574e+00}
};
static const double Coef_m4[3][4] = {
    {-1.653173201148335e-06, 1.036578627632170e-02, 
     2.349390607953272e-02, 1.432904956685477e+00},
    {-8.298537364426537e-07, 5.017456263052972e-03, 
     5.123168011502721e-02, 6.453910704667408e+00},
    {-1.431525987300163e-06, 8.386323466104711e-03, 
     -1.841057020139425e-02, -1.410602407670769e+00}
};
static const double Coef_v4[3][4] = {
    {-2.726183914412441e-07, 2.788507874891710e-02, 
     2.777086294607445e-02, 8.369406298984288e-01},
    {1.118379212729684e-06, 2.433214514397419e-02, 
     2.778340896223197e-02, 1.489387981224663e+00},
    {2.197737873275589e-06, 3.186581505796005e-02, 
     3.808382220884354e-02, 1.958805931276004e+00}
};

/*
 * n from 440 to 1599 
 */

static const double Coef_p5[2][4] = {
    {1.034981541036597e-10, -2.445177000398938e-07, 
     0.000000000000000e+00, 1.451229377475864e+00},
    {-2.291586556531707e-10, 5.414543692806514e-07, 
     0.000000000000000e+00, 3.216167113242079e+00}
};
static const double Coef_m5[2][4] = {
    {-6.578325435644067e-08, 1.648723149067166e-03, 
     1.594968525045459e-02, 5.566082591106806e+00},
    {-6.292364160498604e-08, 1.618047470065775e-03, 
     -7.091699113800587e-03, -2.516741952410371e+00}
};
static const double Coef_v5[2][4] = {
    {-2.802162650788337e-09, 4.051503597935380e-03, 
     4.018981069179972e-03, 9.654061278849895e-01},
    {3.776558110733883e-08, 5.022619018941299e-03, 
     5.525253413878772e-03, 1.450507513327352e+00}
};

/*
 * n from 1600 to 10000 
 */
static const double Coef_p6[2][4] = {
    {-1.586037487404490e-13, 3.575996226727867e-09, 
     0.000000000000000e+00, 2.228310599179340e+00},
    {1.291237745205579e-13, -2.911316152726367e-09, 
     0.000000000000000e+00, 1.814126328168031e+00}
};
static const double Coef_m6[2][4] = {
    {-2.419956255676409e-09, 3.245753451748892e-04, 
     1.895335618211674e-03, 3.388553853864067e+00},
    {-2.419411092563945e-09, 3.245669014250788e-04, 
     -2.327930564510444e-03, -4.162274939236667e+00}
};
static const double Coef_v6[2][4] = {
    {-6.024563976875348e-11, -6.540694956580495e-04, 
     -6.582951415419203e-04, 1.006399508694657e+00},
    {5.024317053887777e-10, 8.898044793516080e-04, 
     9.246987493760628e-04, 1.149073788967684e+00}
};

/*
 * n from 10000 to 30000 
 */
static const double Coef_p7[2][4] = {
    {-1.663426552872397e-14, 1.141056828884990e-09, 
     0.000000000000000e+00, 2.228285989630589e+00},
    {1.354267905471566e-14, -9.289835742028532e-10, 
     0.000000000000000e+00, 1.814142639751731e+00}
};
static const double Coef_m7[2][4] = {
    {-8.929405559006038e-11, 6.319814700961324e-05, 
     4.785131212048377e-04, 4.271922830906078e+00},
    {-8.931137480031157e-11, 6.320244393309693e-05, 
     -5.877524860249395e-04, -5.247218808668549e+00}
};
static const double Coef_v7[2][4] = {
    {-1.418731402291282e-12, -5.512224505288543e-06, 
     -5.638714069888806e-06, 1.006201804172733e+00},
    {1.576782807097003e-11, 1.914006058179041e-04, 
     1.959753272178233e-04, 1.087101027065273e+00}
};

/*
 * Approximate mixture parameters using a rational function that is
 * quadratic in the numerator and linear in the denominator.  The
 * coefficients are in an nrow x 4 array. If the elements of a row
 * are a,b,c,d then the rational approximation is:
 *
 * (a * n^2 + b * n + 1)/(c * n + d)
 *
 */
static void rational_approx(double n, const double (*coef)[4], int nrow, 
			    double *out)
{
    for (int i = 0; i < nrow; i++) {
	double num = coef[i][0] * n * n + coef[i][1] * n + 1;
	double denom = coef[i][2] * n + coef[i][3];
	out[i] = num / denom;
    }
}

namespace glm {

    LGMix::LGMix(double n)
	: _nlast(n), _r(0), _ncomp(0)
    {
	if (n > 0) 
	    updateN(n);
    }

    /*
     * Exact mixture parameters for all integer n less than 20
     */
    void LGMix::updateNExact(int n)
    {
	if (n < 5) {
	    _ncomp = 10;
	    n -= 1;
	    copy(P10[n], P10[n] + _ncomp, _weights);
	    copy(M10[n], M10[n] + _ncomp, _means);
	    copy(V10[n], V10[n] + _ncomp, _variances);
	}
	else {
	    _ncomp = 9;
	    n -= 5;
	    copy(P9[n], P9[n] + _ncomp, _weights);
	    copy(M9[n], M9[n] + _ncomp, _means);
	    copy(V9[n], V9[n] + _ncomp, _variances);
	}
    }

    /*
     * Approximate mixture parameters for n >= 20 using rational functions
     */
    void LGMix::updateNApprox(double n)
    {
	//Upper limit of range of n supported by each approximation
	const int upper[5] = {50, 440, 1600, 10000, 30000};
	//Number of components in each approximation
	const int ncomp[5] = {4, 3, 2, 2, 2};

	//Each of P,M,V is a 3-dimensional ragged array.
	//C array syntax doesn't make this easy
	const double (*P[5])[4] = {Coef_p3, Coef_p4, Coef_p5, Coef_p6, Coef_p7};
	const double (*M[5])[4] = {Coef_m3, Coef_m4, Coef_m5, Coef_m6, Coef_m7};
	const double (*V[5])[4] = {Coef_v3, Coef_v4, Coef_v5, Coef_v6, Coef_v7};

	// Find appropriate range and calculate approximate coefficients
	for (int r = 0; r < 5; r++) {
	    if (n < upper[r]) {
		rational_approx(n, P[r], ncomp[r], _weights);
		rational_approx(n, M[r], ncomp[r], _means);
		rational_approx(n, V[r], ncomp[r], _variances);
		_ncomp = ncomp[r];
		return;
	    }
	}

	//Single component for n > 30000
	_weights[0] = 1;
	_means[0] = 0;
	_variances[0] = 1;
	_ncomp = 1;
    }

    void LGMix::updateN(double n)
    {
	if (n <= 0) {
	    throwLogicError("n out of range in LGMix::updateN");
	}
	else if (n < 20) {
	    int nr = static_cast<int>(n);
	    if (nr != n) {
		throwLogicError("Invalid in in LGMix::updateN");
	    }
	    updateNExact(nr);
	}
	else {
	    updateNApprox(n);
	}
	
	// Rescale by mean and standard deviation of the negative
	// log-gamma distribution
	double mu = -digamma(n);
	double sigma2 = trigamma(n);
	double sigma = sqrt(sigma2);
	    
	for (int i = 0; i < _ncomp; i++) {
	    _means[i] = _means[i] * sigma + mu;
	    _variances[i] *= sigma2;
	}
	_nlast = n;

    }

    void LGMix::update(double z, double n, RNG *rng)
    {
	// Check that value of n has not changed since last update
	if (n != _nlast) {
	    updateN(n);
	}

	vector<double> p(_ncomp);
	//Log probabilities
	double maxp = 0;
	for (int i = 0; i < _ncomp; i++) {
	    p[i] = dnorm(z, _means[i], sqrt(_variances[i]), true) + 
		log(_weights[i]);
	    if (i == 0 || p[i] > maxp)
		maxp = p[i];
	}

	//Cumulative probabilities
	double sump = 0;
	for (int i = 0; i < _ncomp; i++) {
	    p[i] = sump + exp(p[i] - maxp);
	    sump = p[i];
	}
    
	//Sample _r from cumulative probabilities
	double u = rng->uniform() * sump;
	for (_r = 0; _r < _ncomp - 1; _r++) {
	    if (u < p[_r]) {
		break;
	    }
	}
    }

    double LGMix::mean() const
    {
	return _means[_r];
    }

    double LGMix::precision() const
    {
	return 1/_variances[_r];
    }
}
