/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggshell.h"


#include "icontrolmodule.h"
#include "icontrolscript.h"
#include "iedition.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ieventobserver.h"
#include "ifile.h"
#include "ioutputchannel.h"
#include "iversion.h"
#include "iviewmodule.h"

#include "iggdialog.h"
#include "iggextensionwindow.h"
#include "iggframe.h"
#include "iggmainwindow.h"
#include "iggrenderwindow.h"
#include "iggwidgetarea.h"
#include "iggwidgetmisc.h"
#include "iggwidgetprogressbar.h"

#include "ibgdialogsubject.h"
#include "ibgextensionwindowsubject.h"
#include "ibgframesubject.h"
#include "ibgmenuwindowsubject.h"
#include "ibgrenderwindowsubject.h"
#include "ibgshellsubject.h"
#include "ibgwindowhelper.h"

#if defined(I_DEBUG)
#include <vtkTimerLog.h>
#endif

#include "iggsubjectfactory.h"


using namespace iParameter;
using namespace iParameter;


class iggDialogFlashWindow : public iggDialog
{

public:

	iggDialogFlashWindow(iShell *shell) : iggDialog(shell,DialogFlag::Unattached|DialogFlag::Blocking|DialogFlag::NoTitleBar,0,"",0,3,0)
	{
		mFrame->AddLine(new iggWidgetTextArea("%b%+   IFrIT - Ionization FRont Interactive Tool   ",mFrame),3);
		mFrame->AddLine(new iggWidgetTextArea("Version "+iVersion::GetVersion(),mFrame),3);
		mFrame->AddLine(new iggWidgetTextArea(iEdition::GetEditionName(),mFrame),3);

		mFlipper = new iggWidgetLogoFlipper(mFrame);
		mFrame->AddLine(0,mFlipper,0);

		mMessage = new iggWidgetTextArea("Starting...",mFrame);
		mFrame->AddLine(mMessage,3);

		this->ResizeContents(100,100);
	}

	virtual void Show(bool s)
	{
		mFlipper->Reset();
		iggDialog::Show(s);
	}

	void OnInitAtom()
	{
		mFlipper->Advance();
	}

	void PlaceOnDesktop(int dw, int dh)
	{
		int wg[4];
		mSubject->GetWindowGeometry(wg,true);
		wg[0] = (dw-wg[2]-20)/2;
		wg[1] = (dh-wg[3]-20)/2;
		mSubject->SetWindowGeometry(wg);
	}

	void DisplayMessage(const iString &text)
	{
		mMessage->SetText(text);
	}

protected:

	virtual bool CanBeClosed()
	{
		return false; // cannot be closed
	}

	iggWidgetTextArea *mMessage;
	iggWidgetLogoFlipper *mFlipper;
};


class iggOutputChannel : public iOutputChannel
{

public:

	iggOutputChannel(iggMainWindow *mw) : iOutputChannel(mw->GetShell())
	{
		mMainWindow = mw;
	}

protected:

	virtual void DisplayBody(iConsole::MessageType type, const iString &message, const char *file, int line)
	{
		iString m(message);
		if(file != 0) m += iString("\nFile: ") + file + "\nLine: " + iString::FromNumber(line);

		switch(type)
		{
		case iConsole::_Info:
			{
				mMainWindow->WriteToLog("",message);
				break;
			}
		case iConsole::_Warning:
			{
				mMainWindow->PopupWindow(message,iParameter::PopupWindow::Message);
				break;
			}
		case iConsole::_LowError:
			{
				mMainWindow->WriteToLog("Error",m,iColor(255,0,0)); 
				break; 
			}
		case iConsole::_HighError:
			{
				mMainWindow->PopupWindow(m,iParameter::PopupWindow::Warning);
				break;
			}
		case iConsole::_FatalError:
			{
				mMainWindow->PopupWindow(m,iParameter::PopupWindow::Error);
				break;
			}
		case iConsole::_Notification:
			{
				mMainWindow->PopupWindow(m,iParameter::PopupWindow::Warning);
				break;
			}
		}
	}

private:

	iggMainWindow *mMainWindow;
};


iggShell::iggShell(const iString &type, int argc, char **argv) : iShell(type,argc,argv)
{
	mSubject = iggSubjectFactory::CreateShellSubject(this,argc,argv);
	if(mSubject == 0) exit(1);

	//
	//  Default settings
	//
	mShowFlashWindow = true;
	mAccept8BitDisplay = mStartDocked = false;

	mConditionFlags = 0;

	mMainWindow = 0; // very important!!! This tells other components that the main
					 // window does not exist yet.
	mFlashWindow = 0;
}


void iggShell::Exit()
{
	mSubject->Exit();
}


void iggShell::GetDesktopDimensions(int &w, int &h) const
{
	mSubject->GetDesktopDimensions(w,h);
}


void iggShell::DefineSpecificCommandLineOptions()
{
	//
	//  Define generic GUI-specific command-line options
	//
	if(mSubject->CanRecognize8BitDisplay()) this->AddCommandLineOption("8",false,"accept an 8-bit display");
	if(mSubject->CanDockWindows()) this->AddCommandLineOption("d",false,"start IFrIT with windows docked");
	if(mSubject->CanChangeFontSize()) this->AddCommandLineOption("fs",true,"increase/decrease window font size");
	this->AddCommandLineOption("nf",false,"do not show the splash screen at start-up");

	this->AddCommandLineOption("owm",false,"use old-style window manager");
	this->AddCommandLineOption("sds",false,"assume the desktop size is small");
	this->AddCommandLineOption("src",false,"minimize updates for running on a slow remote connection");
}


iggShell::~iggShell()
{
}


void iggShell::PrepareForConstruction()
{
	//
	//  Check for the 8-bit display
	//
	if(!mAccept8BitDisplay && mSubject->CanRecognize8BitDisplay() && mSubject->CanRecognize8BitDisplay(true))
	{
		iConsole::Display(iConsole::_FatalError,"You are not using a True Color (24-bit) monitor.\n"
			"Visualizations will be crappy without the True Color support.\n"
			"If you wish to continue anyway, specify -8 as an option.");
	}

#ifdef I_DEBUG
	iConsole::PrintDebugMessage("Starting...");
#endif
	//
	//  Create and show message window
	//
	if(mShowFlashWindow)
	{
		mFlashWindow = new iggDialogFlashWindow(this); IERROR_ASSERT(mFlashWindow);
		//
		//  calculate the position of the flash window
		//
		int dw, dh;
		mSubject->GetDesktopDimensions(dw,dh);
		//
		//  Allow for the two-monitor desktop
		//
		if(dw > 2*dh)
		{
			dw /= 2;
		}
		mSubject->ProcessEvents(true);
		mFlashWindow->PlaceOnDesktop(dw,dh);
		mFlashWindow->Show(true);
		mSubject->ProcessEvents(true);
		mFlashWindow->DisplayMessage("Creating visualization window...");
	}
}


void iggShell::ConstructorBody()
{
	if(mShowFlashWindow)
	{
		mFlashWindow->DisplayMessage("Creating widgets...");
		mSubject->ProcessEvents(false);
	}

	mMainWindow = new iggMainWindow(this); IERROR_ASSERT(mMainWindow);
	iOutputChannel::SetInstance(new iggOutputChannel(mMainWindow));
	mMainWindow->StartInitialization();
}

  
void iggShell::DestructorBody()
{
	//
	//  Hide windows to avoid extra updates
	//
	mMainWindow->ShowAll(false);
	mMainWindow->Block(true); // block all events
	mMainWindow->Delete();
	mMainWindow = 0;  // tell other objects that the MainWindow does not exist any more
}


void iggShell::PrepareToLoadStateFile()
{
	if(mShowFlashWindow)
	{
		mFlashWindow->DisplayMessage("Loading the state file...");
		mSubject->ProcessEvents(false);
	}
}


void iggShell::Start()
{
	const int topOffset0 = 64;
	const int frameXoffDefault = 13;
	const int frameYoffDefault = 25;
	int frameXoff = 1, frameYoff = 1;
	int topOffset, leftOffset = 0;

#ifdef I_DEBUG
	vtkTimerLog *debugTimer = vtkTimerLog::New(); IERROR_ASSERT(debugTimer);
	debugTimer->StartTimer();
#endif

	if(mStartDocked) mMainWindow->mDocked = true;
	mMainWindow->PreShowInitialization();
	if(mStartDocked) mMainWindow->mDocked = false;
	mSubject->ProcessEvents(false);
		
	if(mShowFlashWindow)
	{
		mFlashWindow->DisplayMessage("Showing windows...");
		mSubject->ProcessEvents(false);
	}

	//
	//  Show visualization windows
	//
	if(!this->HasStateFile() && !this->IsDesktopSmall(false))
	{
		//
		// Calculate the left offset of the visualization window (it is the only one)
		//
		int dw, dh;
		mSubject->GetDesktopDimensions(dw,dh);
		//
		//  Allow for the two-monitor desktop
		//
		if(dw > 2*dh)
		{
			dw /= 2;
		}

		int mwg[4], vwg[4];
		iggRenderWindow *visWindow = iRequiredCast<iggRenderWindow>(INFO,this->GetControlModule()->GetViewModule()->GetRenderWindow());

		//
		//  squeeze windows to the minimum possible size
		//
		mwg[0] = mwg[1] = mwg[2] = mwg[3] = 1;
		mMainWindow->GetSubject()->SetWindowGeometry(mwg);
		mMainWindow->GetExtensionWindow()->GetSubject()->SetWindowGeometry(mwg);
		mSubject->ProcessEvents(false);

		topOffset = this->IsDesktopSmall(true) ? 0 : topOffset0;
		//
		//  Show all windows
		//
		if(mStartDocked)
		{
			//
			//  If we start in the docked mode, trust the window manager
			//
			mMainWindow->DockWindows(true,false);
			mMainWindow->GetSubject()->GetWindowGeometry(mwg);

			bool redo = false;
			if(mwg[2] < (4*mwg[3]/3))
			{
				mwg[2] = (4*mwg[3]/3);
				redo = true;
			}

			leftOffset = (dw-mwg[2]-20)/2;
			if(leftOffset < 0) leftOffset = 0;
			if(mwg[0]<leftOffset || mwg[1]<topOffset)
			{
				mwg[0] = leftOffset;
				mwg[1] = topOffset;
				redo = true;
			}
			if(redo) mMainWindow->GetSubject()->SetWindowGeometry(mwg);

			mMainWindow->ShowAll(true);
		}
		else
		{
			//
			//  Place the windows on the screen
			//
			mMainWindow->GetSubject()->GetWindowGeometry(mwg,true);
			visWindow->GetSubject()->GetWindowGeometry(vwg,true);

			leftOffset = (dw-vwg[2]-mwg[2]-20)/2;
			if(leftOffset < 0) leftOffset = 0;
			//
			//  Move the view module window
			//
			vwg[0] = leftOffset;
			vwg[1] = topOffset;
			visWindow->GetSubject()->SetWindowGeometry(vwg);

			//
			//  Show the visualization window and give window manager a chance to decorate it.
			//			     
			this->GetControlModule()->Render(RenderOption::All);
			mSubject->ProcessEvents(false);

			//
			//  Get the frame geometry
			//
			mSubject->ProcessEvents(true);
			visWindow->GetSubject()->GetFrameSize(frameXoff,frameYoff);
			if(frameXoff == 0) frameXoff = frameXoffDefault; // X11 failed to assign a proper frame
			if(frameYoff == 0) frameYoff = frameYoffDefault; // X11 failed to assign a proper frame

			mwg[0] = vwg[0] + vwg[2] + frameXoff;
			mwg[1] = topOffset;
			mMainWindow->GetSubject()->SetWindowGeometry(mwg,true);
	
			int ewg[4], ewg2[4];
			ewg[0] = leftOffset;
			ewg[1] = vwg[1] + vwg[3] + frameYoff;
			ewg[2] = vwg[2];
			ewg[3] = mwg[3] - vwg[3] - frameYoff;
			mMainWindow->GetExtensionWindow()->GetSubject()->SetWindowGeometry(ewg,true);
			mMainWindow->GetExtensionWindow()->GetSubject()->GetWindowGeometry(ewg2,true);
			//
			//  Stretch main window to match the extension window, but only if the desktop tall enough.
			//
			if(ewg2[3]>ewg[3] && !this->IsDesktopSmall(true))
			{
				mMainWindow->GetSubject()->GetWindowGeometry(mwg,true);
				mwg[3] += (ewg2[3]-ewg[3]);
				mMainWindow->GetSubject()->SetWindowGeometry(mwg,true);
			}

			//
			//  Show the main window and give a chance to the Window Manager to decorate the window
			//			     
			mMainWindow->GetSubject()->Show(true);
			mSubject->ProcessEvents(false);

			mMainWindow->GetExtensionWindow()->GetSubject()->Show(true);
			mSubject->ProcessEvents(false);
		}
	}
	else
	{
		this->GetControlModule()->Render(RenderOption::All);
		mMainWindow->GetSubject()->Show(true);
		mMainWindow->GetExtensionWindow()->GetSubject()->Show(true);
	}
	
	mSubject->CompleteStartUp();

	if(mShowFlashWindow)
	{
		//
		//  Remove the flash window
		//
		mFlashWindow->Show(false);
		delete mFlashWindow;
		mFlashWindow = 0;
	}

	mSubject->ProcessEvents(false);
	mMainWindow->PostShowInitialization();

#ifdef I_DEBUG
	debugTimer->StopTimer();
	iConsole::PrintDebugMessage("Init: "+iString::FromNumber(debugTimer->GetElapsedTime()));
	debugTimer->Delete();
#endif
}


void iggShell::RunBody()
{
	mSubject->EnterEventLoop();
}


bool iggShell::AnalyseOneExtendedCommandLineOption(const struct Option &o)
{
	if(o.Name == "fs")
	{
		int s;
		bool ok;
		if((o.Value[0]=='+' || o.Value[0]=='-') && (s=o.Value.ToInt(ok))!=0 && ok)
		{
			ibgWindowHelper::ChangeFontSize(s);
			return true;
		}
		else
		{
			iConsole::Display(iConsole::_HighError,"Invalid font size specification; should be in the form +<number> or -<number>. Abort.");
			exit(1);
		}
	} 

	if(o.Name == "d")
	{
		mStartDocked = true;
		return true;
	} 

	if(o.Name == "nf")
	{
		mShowFlashWindow = false;
		return true;
	} 

	if(o.Name == "8")
	{
		mAccept8BitDisplay = true;
		return true;
	}

	//
	//  behaviour settings
	//
	if(o.Name == "owm")
	{
		mConditionFlags |= Condition::OldWindowManager;
		return true;
	}

	if(o.Name == "sds")
	{
		mConditionFlags |= Condition::SmallDesktopSize;
		return true;
	} 

	if(o.Name == "src")
	{
		mConditionFlags |= Condition::SlowRemoteConnection;
		return true;
	}
	
	return false;
}


//
//  Running conditions
//
bool iggShell::IsDesktopSmall(bool vert) const
{
	if(this->CheckCondition(Condition::SmallDesktopSize)) return true;

	int w, h;
	mSubject->GetDesktopDimensions(w,h);
	if(vert) return (h < 900); else return (w < 1200);
}


bool iggShell::CheckCondition(int flag) const
{
	if((mConditionFlags & flag) != 0) return true; else return mSubject->CheckCondition(flag);
}


//
//  State saving
//
bool iggShell::LoadShellStateFromFileBody(iFile &F)
{
	iString ws, line;
	//
	//  Restore the state of the main window
	//
	F.Rewind();
	while(F.ReadLine(line)) 
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);

		if(ws == iggMainWindow::Type().FullName()) 
		{
			ws = line.Section(" ",1);
			mMainWindow->UnPackCompleteState(ws);
		}
	}
	return true;
}


bool iggShell::SaveShellStateToFileBody(iFile &F) const
{
	//
	//  Save the state of the main window 
	//
	iString ws;
	mMainWindow->PackCompleteState(ws);
	ws = iggMainWindow::Type().FullName() + " " + ws;
	return F.WriteLine(ws);
}


void iggShell::ProcessEvents(bool sync)
{
	mSubject->ProcessEvents(sync);
}


void iggShell::OnInitAtomBody(bool timed)
{
	if(timed && mFlashWindow!=0) mFlashWindow->OnInitAtom();
}


void iggShell::OnLoadStateFileAtomBody(int prog)
{
	if(mMainWindow!=0 && mMainWindow->GetProgressBar()!=0)
	{
		if(prog == 0) mMainWindow->GetProgressBar()->Reset();
		mMainWindow->GetProgressBar()->SetProgress(prog);
		if(prog == 100) mMainWindow->GetProgressBar()->Reset();
	}
}

#endif
