/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggpagesurface.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "iimagefactory.h"
#include "ishell.h"
#include "isurfaceviewsubject.h"
#include "iviewmodule.h"
#include "iviewobject.h"
#include "iviewobjectfamily.h"

#include "iggdatatypeprovider.h"
#include "iggframecurrentinstance.h"
#include "iggframedatavariablelist.h"
#include "iggframedoublebutton.h"
#include "iggframematerialproperties.h"
#include "iggframeobjectcontrols.h"
#include "iggframepaletteselection.h"
#include "iggframereplicate.h"
#include "iggmainwindow.h"
#include "iggwidgetkeybutton.h"
#include "iggwidgetkeycolorselection.h"
#include "iggwidgetkeyselectionbox.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetotherbutton.h"

#include "ibgframesubject.h"
#include "ibgwidgetareasubject.h"
#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgethelper.h"
#include "ibgwidgetselectionboxsubject.h"

#include "iggsubjectfactory.h"

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


using namespace iParameter;
using namespace iParameter;


namespace iggPageSurface_Private
{
	//
	//  Helper classes
	//
	class MethodRadioBox : public iggWidgetKeyRadioBox
	{
		
	public:
		
		MethodRadioBox(const iObjectKey &key, iggFrame *parent) : iggWidgetKeyRadioBox(1,"Method",0,key,parent)
		{
			mFlipBuddy = 0;
			mObjectBuddy = 0;

			this->InsertItem("Isosurface");
			this->InsertItem("Sphere");
			this->InsertItem("Plane");
		}

		void SetFlipBuddy(iggFrameFlip *b)
		{
			mFlipBuddy = b;
		}

		void SetObjectBuddy(iggFrameObjectControls *b)
		{
			mObjectBuddy = b;
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->iggWidgetKeyRadioBox::UpdateWidgetBody();
			this->UpdateBuddy();
		}
		
		virtual void OnInt1Body(int i)
		{
			this->UpdateBuddy();
			iggWidgetKeyRadioBox::OnInt1Body(i);
		}

		virtual void UpdateBuddy()
		{
			int i = mSubject->GetValue();

			if(mFlipBuddy != 0)
			{
				if(i == 0)
				{
					mFlipBuddy->ShowLayer(0);
				}
				else
				{
					mFlipBuddy->ShowLayer(1);
				}
			}

			if(mObjectBuddy == 0) return;

			switch(i)
			{
			case 0:
				{
					break;
				}
			case 1:
				{
					mObjectBuddy->EnableDirection(false);
					mObjectBuddy->EnableSize(true);
					break;
				}
			case 2:
				{
					mObjectBuddy->EnableDirection(true);
					mObjectBuddy->EnableSize(false);
					break;
				}
			default:
				{
					IERROR_LOW("Incorrect button in MethodRadioBox.");
				}
			}
		}

		iggFrameFlip *mFlipBuddy;
		iggFrameObjectControls *mObjectBuddy;
	};
	

	class SideRadioBox : public iggWidget
	{
		
	public:
		
		SideRadioBox(iggFrame *parent) : iggWidget(parent)
		{
			mVarBuddy = 0;
			mPalBuddy = 0;

			mSubject = iggSubjectFactory::CreateWidgetRadioBoxSubject(this,1,"Side");
			mSubject->InsertItem("Exterior");
			mSubject->InsertItem("Interior");

			mSubject->SetValue(0);

			this->SetBaloonHelp("Sets the side of the surface to paint.","A surface can be painted separately on both sides. Because OpenGL cannot paint the surface from two different sides, IFrIT replicates the surface if it needs to paint both sides. That means that by painting the interior side of the surface, you double the number of polygons. ");
		}

		void SetVarBuddy(iggFrameDataVariableList *b)
		{
			mVarBuddy = b;
		}

		void SetPalBuddy(iggFramePaletteSelection *b)
		{
			mPalBuddy = b;
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->UpdateBuddy();
		}
		
		virtual void OnInt1Body(int)
		{
			this->UpdateBuddy();
		}

		virtual void UpdateBuddy()
		{
			int i = mSubject->GetValue();

			if(i>=0 && i<=1)
			{
				if(mVarBuddy != 0)
				{
					mVarBuddy->SetIndex(i);
				}
				if(mPalBuddy != 0)
				{
					mPalBuddy->SetIndex(i);
				}
			}
		}

		ibgWidgetRadioBoxSubject *mSubject;
		iggFrameDataVariableList *mVarBuddy;
		iggFramePaletteSelection *mPalBuddy;
	};


	class BatchCreateFrame : public iggFrame
	{
		
	public:

		BatchCreateFrame(iggFrame *parent) : iggFrame("Interpolate between instances",parent,2)
		{
			//
			//  Subjects are inserted into a parent frame layout, so we need to create a child frame
			//  which is then used to create subjects. This design is somewhat contrived, but it
			//  avoid a need to create at least 3 new widgets
			//
			iggFrame *frame = new iggFrame(this,1);
			this->AddLine(frame,2);

			mSubject->PlaceWidget(0,1,iggSubjectFactory::CreateWidgetDisplayAreaSubject(frame,"Instance 1"),1,true);
			mSpinBox1 = iggSubjectFactory::CreateWidgetSpinBoxSubject(frame,1,1,"",1);
			mSubject->PlaceWidget(1,1,mSpinBox1,1,true);

			mSubject->PlaceWidget(0,2,iggSubjectFactory::CreateWidgetDisplayAreaSubject(frame,"Instance 2"),1,true);
			mSpinBox2 = iggSubjectFactory::CreateWidgetSpinBoxSubject(frame,1,1,"",1);
			mSubject->PlaceWidget(1,2,mSpinBox2,1,true);
			
			mSubject->PlaceWidget(0,3,iggSubjectFactory::CreateWidgetDisplayAreaSubject(frame,"Number to create"),1,true);
			mSpinBox3 = iggSubjectFactory::CreateWidgetSpinBoxSubject(frame,1,100,"",1);
			mSubject->PlaceWidget(1,3,mSpinBox3,1,true);

			mSubject->PlaceWidget(0,4,iggSubjectFactory::CreateWidgetDisplayAreaSubject(frame,""),1,true);
			mButton = iggSubjectFactory::CreateWidgetButtonSubject(frame,ButtonType::PushButton,"Create",1);
			mSubject->PlaceWidget(1,4,mButton,1,true);

			this->SetColStretch(0,10);

			this->SetBaloonHelp("Creates multiple surfaces by interpolating between the two other instances. Press Shift+F1 for more help.","This control allows you to create several (up to 100) surfaces at once, by interpolating between two existing different instances. The two instances must be compatible with each other: they must use the same method (for example, both being isosurfaces of the same scalar field variable, or both being spheres, etc) and must both use the same data channel. During the interpolation, surface opacities, isosurface leveles, colors, sphere and plane positions, sphere sizes, and plane directions are all interpolated linear between the two specified instances. The batch creation may be useful for, for example, creating a set of semi-transparent isosurface as a surrogate for volume rendering.");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			int n;

			if(!this->GetShell()->GetControlModule()->QueryValue(iViewModule::KeySurfaceMax(),n))
			{
				this->Enable(false);
				return this->UpdateFailed();
			}

			n++;
			if(mSpinBox1->Count() != n) mSpinBox1->SetRange(1,n);
			if(mSpinBox2->Count() != n) mSpinBox2->SetRange(1,n);

			if(n>1 && mSpinBox1->GetValue()==mSpinBox2->GetValue())
			{
				//
				//  Make two spin boxes different
				//
				if(mSpinBox1->GetValue() < n)
				{
					mSpinBox2->SetValue(mSpinBox1->GetValue()+1);
				}
				else
				{
					mSpinBox2->SetValue(1);
				}
			}

			this->Enable(this->AreTwoInstancesConsistent());
		}
		
		bool AreTwoInstancesConsistent() const
		{
			int m1, m2, v1, v2, dc1, dc2;

			//
			//  Split one if for debugging
			//
			if(mSpinBox1->GetValue() == mSpinBox2->GetValue()) return false;
			dc1 = this->GetShell()->GetControlModule()->GetViewModule()->GetViewObjectFamily(ViewSubject::Id::Surface)->GetMember(mSpinBox1->GetValue()-1)->GetActiveDataTypeIndex();
			dc2 = this->GetShell()->GetControlModule()->GetViewModule()->GetViewObjectFamily(ViewSubject::Id::Surface)->GetMember(mSpinBox2->GetValue()-1)->GetActiveDataTypeIndex();
				
			if(!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyMethod(),m1,-1,mSpinBox1->GetValue()-1)) return false;
			if(!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyMethod(),m2,-1,mSpinBox2->GetValue()-1)) return false;
			if(!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyIsoSurfaceVar(),v1,-1,mSpinBox1->GetValue()-1)) return false;
			if(!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyIsoSurfaceVar(),v2,-1,mSpinBox2->GetValue()-1)) return false;

			return (m1==m2) && (m1!=SurfaceMethod::IsoSurface || v1==v2) && (dc1==dc2);
		}

		virtual void OnInt1Body(int)
		{
			this->UpdateWidget();
		}

		virtual void OnVoid1Body()
		{
			int i, n, nmax, m, v, dc;
			iColor c, cmin, cmax; float cstp[3];
			float lmin, lmax, lstp, l;
			float omin, omax, ostp, o;
			double pmin[3], pmax[3], pstp[3], p[3];
			double smin, smax, sstp, s;
			float dmin[3], dmax[3], dstp[3], d[3];
			
			if(!this->AreTwoInstancesConsistent())
			{
				IERROR_LOW("Two isosurafces instances are not compatible or are not isosurfaces.");
				mWidgetHelper->Enable(false);
				return;
			}

			dc = this->GetShell()->GetControlModule()->GetViewModule()->GetViewObjectFamily(ViewSubject::Id::Surface)->GetMember(mSpinBox1->GetValue()-1)->GetActiveDataTypeIndex();

			if(
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyMethod(),m,-1,mSpinBox1->GetValue()-1) ||
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyIsoSurfaceVar(),v,-1,mSpinBox1->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyIsoSurfaceLevel(),lmin,-1,mSpinBox1->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyIsoSurfaceLevel(),lmax,-1,mSpinBox2->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyPosition(true),pmin,3,-1,mSpinBox1->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyPosition(true),pmax,3,-1,mSpinBox2->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeySize(true),smin,-1,mSpinBox1->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeySize(true),smax,-1,mSpinBox2->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyPlaneDirection(),dmin,3,-1,mSpinBox1->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyPlaneDirection(),dmax,3,-1,mSpinBox2->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyOpacity(),omin,-1,mSpinBox1->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyOpacity(),omax,-1,mSpinBox2->GetValue()-1) ||
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyColor(),cmin,-1,mSpinBox1->GetValue()-1) || 
				!this->GetShell()->GetControlModule()->QueryValue(iSurfaceViewSubject::KeyColor(),cmax,-1,mSpinBox2->GetValue()-1)) 
			{
				IERROR_LOW("Unable to query Surface object.");
				mWidgetHelper->Enable(false);
				return;
			}
		
			nmax = mSpinBox3->GetValue();

			ostp = (omax-omin)/(nmax+1);
			cstp[0] = (float)(cmax.Red()-cmin.Red())/(nmax+1.0f);
			cstp[1] = (float)(cmax.Green()-cmin.Green())/(nmax+1.0f);
			cstp[2] = (float)(cmax.Blue()-cmin.Blue())/(nmax+1.0f);
			lstp = (lmax-lmin)/(nmax+1);
			sstp = (smax-smin)/(nmax+1);
			for(i=0; i<3; i++)
			{
				pstp[i] = (pmax[i]-pmin[i])/(nmax+1);
				dstp[i] = (dmax[i]-dmin[i])/(nmax+1);
			}

			if((m==SurfaceMethod::IsoSurface) && (fabs(lstp)<0.01))
			{
				IERROR_LOW("Cannot create new isosurface: the two limits are too close.");
				mWidgetHelper->Enable(false);
				return;
			}
						
			for(n=0; n<nmax; n++)
			{
				o = omin + ostp*(n+1);
				c = iColor(round(cmin.Red()+cstp[0]*(n+1.0)),round(cmin.Green()+cstp[1]*(n+1.0)),round(cmin.Blue()+cstp[2]*(n+1.0)));
				l = lmin + lstp*(n+1);
				s = smin + sstp*(n+1);
				for(i=0; i<3; i++)
				{
					p[i] = pmin[i] + pstp[i]*(n+1);
					d[i] = dmin[i] + dstp[i]*(n+1);
				}

				if(this->GetShell()->GetControlModule()->CreateObject(iSurfaceViewSubject::Type(),false))
				{
					this->GetShell()->GetControlModule()->GetViewModule()->GetViewObjectFamily(ViewSubject::Id::Surface)->GetCurrentMember()->SetActiveDataTypeIndex(dc);
					//
					//  Set the parameters of the new instance
					//
					iString ws;
					this->GetShell()->GetControlModule()->PackCommand(ws,iSurfaceViewSubject::KeyDataType(),dc);
					this->GetShell()->GetControlModule()->PackCommand(ws,iSurfaceViewSubject::KeyMethod(),m);
					this->GetShell()->GetControlModule()->PackCommand(ws,iSurfaceViewSubject::KeyOpacity(),o);
					this->GetShell()->GetControlModule()->PackCommand(ws,iSurfaceViewSubject::KeyColor(),c);
					this->GetShell()->GetControlModule()->PackCommand(ws,iSurfaceViewSubject::KeyIsoSurfaceVar(),v);
					this->GetShell()->GetControlModule()->PackCommand(ws,iSurfaceViewSubject::KeyIsoSurfaceLevel(),l);
					this->GetShell()->GetControlModule()->PackCommand(ws,iSurfaceViewSubject::KeyPosition(true),p,3);
					this->GetShell()->GetControlModule()->PackCommand(ws,iSurfaceViewSubject::KeySize(true),s);
					this->GetShell()->GetControlModule()->PackCommand(ws,iSurfaceViewSubject::KeyPlaneDirection(),d,3);
					this->GetShell()->GetControlModule()->Execute(ws,false,this->GetExecuteFlags());
				}
			}
			
			this->GetShell()->GetControlModule()->Render();
		}

		ibgWidgetSpinBoxSubject *mSpinBox1, *mSpinBox2, *mSpinBox3;
		ibgWidgetButtonSubject *mButton;
	};


	class PrintInfoButton : public iggWidgetSimpleButton
	{

	public:

		PrintInfoButton(iggFrame *parent) : iggWidgetSimpleButton("Print info",parent)
		{
			this->SetBaloonHelp("Print some information about the current surface");
		}

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);

			iString ws;
			this->GetShell()->GetControlModule()->PackCommand(ws,iSurfaceViewSubject::KeyOutputProperties(),true);
			this->GetShell()->GetControlModule()->Execute(ws,false,this->GetExecuteFlags());

			this->GetMainWindow()->Block(false);
		}
	};
};


using namespace iggPageSurface_Private;


iggPageSurface::iggPageSurface(iggFrameBase *parent) : iggPageObject(parent,ViewSubject::Id::Surface)
{
	const iImage *icon = iImageFactory::FindIcon("surf.png");

	//
	//  Main page
	// ************************************************
	//
	iggFrame *page0 = new iggFrame(mBook,1);
	mBook->AddPage("Main",icon,page0);
	//
	//  Top half-page sub-frame
	//
	iggFrame *page0top = new iggFrame(page0,4);
	page0->AddLine(page0top);
	//
	//  Show
	//
	iggWidget *show = new iggWidgetShowButton(iSurfaceViewSubject::Type(),page0top);
	iggFrameCurrentInstance *ci1 = new iggFrameCurrentInstance(false,"Instance","Instance",iViewModule::KeySurfaceCurrent(),iViewModule::KeySurfaceMax(),page0top);
	ci1->AddDependent(this);
	page0top->AddLine(show,static_cast<iggWidget*>(0),static_cast<iggWidget*>(0),ci1);
	page0top->AddSpace(2);
	//
	//  Method & flip normals
	//
	iggFrame *mf = new iggFrame(page0top,2);
	MethodRadioBox *mb = new MethodRadioBox(iSurfaceViewSubject::KeyMethod(),mf);
	iggFrame *mf2 = new iggFrame(mf,1);
	mf2->AddLine(new iggWidgetKeyColorSelection(iSurfaceViewSubject::KeyColor(),mf2,false,0));
	mf2->AddLine(new iggWidgetKeyCheckBox("Flip normals",iSurfaceViewSubject::KeyNormalsFlipped(),mf2));
	mf->AddLine(mb,mf2);
	mf2 = new iggFrame(page0top);
	mf2->AddSpace(1);
	mf2->AddLine(new PrintInfoButton(mf2));
	mf2->AddSpace(1);
	page0top->AddLine(mf,2,mf2,1);
	page0top->AddSpace(2);
	//
	//  Opacity
	//
	iggWidgetKeyFloatSlider *os = new iggWidgetKeyFloatSlider(0.0,1.0,100,0,5,"Opacity",iSurfaceViewSubject::KeyOpacity(),RenderMode::UseGlobal,page0top);
	page0top->AddLine(os,3);
	iggWidgetAllInstancesCheckBox *li = new iggWidgetAllInstancesCheckBox(page0top);
	li->AddDependent(os);
	page0top->AddLine(li,2);
	page0top->AddSpace(10);

	page0top->SetColStretch(2,3);

	//
	//  Lower half-page: various method-dependent settings in a headless mBook
	//
	iggFrame *page0bot = new iggFrame(page0,1);
	page0->AddLine(page0bot);

	iggFrameFlip *msflip = new iggFrameFlip(page0bot);
	page0bot->AddLine(msflip);
	page0bot->SetColStretch(0,10);
//	page0bot->SetColStretch(1,1);
	mb->SetFlipBuddy(msflip);

	//
	//  Isosurface settings
	//
	iggFrame *mspage0 = new iggFrame(msflip,4);
	msflip->AddLayer(mspage0);

	iggFrame *iv = new iggFrame(mspage0,1);
	iggFrameDataVariableList *io = new iggFrameDataVariableList(mProvider,"Isosurface of...",iSurfaceViewSubject::KeyIsoSurfaceVar(),0,iv);
	io->Complete();
	io->AddDependent(show);
	iv->AddLine(io);
	iv->AddLine(new iggWidgetKeyCheckBox("Optimize",iSurfaceViewSubject::KeyIsoSurfaceOptimization(),iv));
	
	iggWidgetKeyRadioBox *id = new iggWidgetKeyRadioBox(1,"Reduction",0,iSurfaceViewSubject::KeyIsoSurfaceReduction(),mspage0);
	id->InsertItem("None");
	id->InsertItem("Light");
	id->InsertItem("Medium");
	id->InsertItem("Heavy");

	iggWidgetKeyRadioBox *is = new iggWidgetKeyRadioBox(1,"Smoothing",0,iSurfaceViewSubject::KeyIsoSurfaceSmoothing(),mspage0);
	is->InsertItem("None");
	is->InsertItem("Light");
	is->InsertItem("Medium");
	is->InsertItem("Heavy");

	mspage0->AddLine(iv,id,is);
	mspage0->AddLine(new iggWidgetKeyCheckBox("Use alternative reduction method",iSurfaceViewSubject::KeyAlternativeIsoSurfaceReductionMethod(),mspage0),3);
	
	mspage0->AddSpace(10);

	iggWidget *il = new iggWidgetKeyVariableLimitsSlider(mProvider,5,"Level",iSurfaceViewSubject::KeyIsoSurfaceLevel(),&iSurfaceViewSubject::KeyIsoSurfaceVar(),0,RenderMode::UseGlobal,mspage0);
	io->AddDependent(il);
	mspage0->AddLine(il,3);
	
	mspage0->AddSpace(10);
	mspage0->SetColStretch(3,10);

	//
	//  Sphere & plane settings
	//
	iggFrame *mspage1 = new iggFrame(msflip);
	msflip->AddLayer(mspage1);

	iggFrameObjectControls *sc= new iggFrameObjectControls("",iSurfaceViewSubject::KeyPosition(true),iSurfaceViewSubject::KeyPlaneDirection(),iSurfaceViewSubject::KeySize(true),mspage1);
	mb->SetObjectBuddy(sc);
	mspage1->AddLine(sc);

	mspage1->AddSpace(10);
	mspage0->SetColStretch(0,10);

	//
	//  Paint page
	// ************************************************
	//
	iggFrame *page1 = new iggFrame(mBook,1);
	mBook->AddPage("Paint",icon,page1);
	//
	//  Book
	//
	iggFrameBook *pb = new iggFrameBook(page1);
	page1->AddLine(pb);
	//
	//  Palette page
	//
	iggFrame *pbpage0 = new iggFrame(pb,3);
	pb->AddPage("Palette",icon,pbpage0);

	iggFrame *pf = new iggFrame(pbpage0,2);
	iggFrameDataVariableList *pl = new iggFrameDataVariableList(mProvider,"Paint with...",iSurfaceViewSubject::KeyPaintVar(),0,pf,0);
	pl->InsertItem("None");
	pl->Complete();

	SideRadioBox *ss = new SideRadioBox(pf);
	ss->SetVarBuddy(pl);

	pf->AddLine(pl,ss);
	pbpage0->AddLine(pf);

	pbpage0->AddSpace(10);

	iggFramePaletteSelection *ps = new iggFramePaletteSelection(true,iSurfaceViewSubject::KeyPalette(),pbpage0,0);
	ss->SetPalBuddy(ps);

	pbpage0->AddLine(ps,2);

	pbpage0->AddSpace(20);

	pbpage0->SetColStretch(1,10);
	pbpage0->SetColStretch(2,3);

	//
	//  Material page
	//
	iggFrame *pbpage1 = new iggFrame(pb,2);
	pb->AddPage("Material",icon,pbpage1);

	iggFrameMaterialProperties *mp = new iggFrameMaterialProperties(false,true,iSurfaceViewSubject::Type(),pbpage1,0);
	pbpage1->AddLine(mp);
	
	pbpage1->AddSpace(10);
	pbpage1->SetColStretch(0,10);
	pbpage1->SetColStretch(1,3);

	//
	//  Instances page
	// ************************************************
	//
	iggFrame *page2 = new iggFrame(mBook,3);
	mBook->AddPage("Instances",icon,page2);
	//
	//  Instance #
	//
	iggFrameCurrentInstance *ci2 = new iggFrameCurrentInstance(false,"Instance","Instance",iViewModule::KeySurfaceCurrent(),iViewModule::KeySurfaceMax(),page2);
	ci2->AddDependent(this);
	page2->AddLine(ci2);
	page2->AddSpace(2);
	//
	//  Color
	//
	page2->AddLine(new iggWidgetKeyColorSelection(iSurfaceViewSubject::KeyColor(),page2,false,0),new iggFrameCreateDeleteButton(this,iSurfaceViewSubject::Type(),true,"instance",page2));
	page2->AddSpace(10);

	page2->AddLine(new BatchCreateFrame(page2),2);
	page2->AddSpace(10);

	page2->SetColStretch(2,10);

	//
	//  Replicate page
	// ************************************************
	//
	iggFrame *page3 = new iggFrame(mBook,2);
	mBook->AddPage("Replicate",icon,page3);
	//
	//  Replicate
	//
	page3->AddLine(new iggFrameReplicate(iSurfaceViewSubject::Type(),page3));
	page3->AddSpace(10);
	page3->SetColStretch(1,3);
}

#endif
