/*
   Copyright (c) 2004, 2005 by AOSASA Shigeru and Red Hat, Inc.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:

   - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

   - Redistributions in binary form must reproduce the above
   copyright notice, this list of conditions and the following
   disclaimer in the documentation and/or other materials provided
   with the distribution.  

   - Neither the name of the AOSASA Shigeru, Red Hat, Inc
   nor the names of its contributors may be used to endorse or
   promote products derived from this software without specific
   prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
   COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
   INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
   (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
   SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
   STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
   OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*
   This software development was supported by
   Information-technorogy Promotion Agency, Japan (IPA).
*/

#include "gvalue-utils.h"

#include <stdlib.h>
#include <string.h>


void
g_value_fill0 (GValue *value)
{
  g_return_if_fail (value != NULL);

  memset (value, 0, sizeof (GValue));
}



static void
g_value_set_char_by_string (GValue      *value,
			    const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  g_value_set_char (value, strtol (string, NULL, 0));
}

static void
g_value_set_uchar_by_string (GValue      *value,
			     const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  g_value_set_uchar (value, strtoul (string, NULL, 0));
}

static void
g_value_set_boolean_by_string (GValue      *value,
			       const gchar *string)
{
  gchar *tmp;

  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  tmp = g_strdup (string);
  tmp = g_strstrip (tmp);

  if (0)
    {
    }
  else if (g_ascii_strcasecmp ("TRUE", tmp) == 0)
    {
      g_value_set_boolean (value, TRUE);
    }
  else if (g_ascii_strcasecmp ("FALSE", tmp) == 0)
    {
      g_value_set_boolean (value, FALSE);
    }

  g_free (tmp);
}

static void
g_value_set_int_by_string (GValue      *value,
			   const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  g_value_set_int (value, strtol (string, NULL, 0));
}

static void
g_value_set_uint_by_string (GValue      *value,
			    const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  g_value_set_uint (value, strtoul (string, NULL, 0));
}

static void
g_value_set_long_by_string (GValue      *value,
			    const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  g_value_set_long (value, strtol (string, NULL, 0));
}

static void
g_value_set_ulong_by_string (GValue      *value,
			     const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  g_value_set_ulong (value, strtoul (string, NULL, 0));
}

static void
g_value_set_int64_by_string (GValue      *value,
			     const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  g_value_set_int64 (value, strtoll (string, NULL, 0));
}

static void
g_value_set_uint64_by_string (GValue      *value,
			      const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  g_value_set_uint64 (value, strtoull (string, NULL, 0));
}

static void
g_value_set_enum_by_string (GValue      *value,
			    const gchar *string)
{
  GEnumClass *enum_class;
  GEnumValue *enum_values;
  guint n_enum_values;
  gchar *tmp;
  guint i;

  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  tmp = g_strdup (string);
  tmp = g_strstrip (tmp);

  enum_class = (GEnumClass *)g_type_class_ref (G_VALUE_TYPE (value));

  enum_values   = enum_class->values;
  n_enum_values = enum_class->n_values;

  for (i = 0; i < n_enum_values; ++i)
    {
      if ((strcmp (enum_values[i].value_name, tmp) == 0) ||
	  (strcmp (enum_values[i].value_nick, tmp) == 0))
	{
	  g_value_set_enum (value, enum_values[i].value);
	  break;
	}
    }

  g_type_class_unref (enum_class);

  g_free (tmp);
}

static void
g_value_set_flags_by_string (GValue      *value,
			     const gchar *string)
{
  gchar **tokens;
  GFlagsClass *flags_class;
  GFlagsValue *flags_values;
  guint n_flags_values;
  gchar *tmp;
  guint i,j;
  guint v;

  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  v = 0;

  tokens = g_strsplit (string, "|", -1);

  flags_class = (GFlagsClass *)g_type_class_ref (G_VALUE_TYPE (value));

  flags_values   = flags_class->values;
  n_flags_values = flags_class->n_values;

  for (j = 0; tokens[j] != NULL; ++j)
    {
      tmp = g_strdup (tokens[j]);
      tmp = g_strstrip (tmp);

      for (i = 0; i < n_flags_values; ++i)
	{
	  
	  if ((strcmp (flags_values[i].value_name, tmp) == 0) ||
	      (strcmp (flags_values[i].value_nick, tmp) == 0))
	    {
	      v |= flags_values[i].value;
	      break;
	    }
	}

      g_free (tmp);
    }

  g_type_class_unref (flags_class);

  g_strfreev (tokens);
}

static void
g_value_set_float_by_string (GValue      *value,
			     const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  g_value_set_float (value, g_strtod (string, NULL));
}

static void
g_value_set_double_by_string (GValue      *value,
			      const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (string != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  g_value_set_double (value, g_strtod (string, NULL));
}



void
g_value_set_by_string (GValue      *value,
		       const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  switch (G_TYPE_FUNDAMENTAL (G_VALUE_TYPE (value)))
    {
    case G_TYPE_INVALID:
    case G_TYPE_NONE:
    case G_TYPE_INTERFACE:
      /* XXX */
      break;

    case G_TYPE_CHAR:
      g_value_set_char_by_string (value, string);
      break;

    case G_TYPE_UCHAR:
      g_value_set_uchar_by_string  (value, string);
      break;

    case G_TYPE_BOOLEAN:
      g_value_set_boolean_by_string (value, string);
      break;

    case G_TYPE_INT:
      g_value_set_int_by_string (value, string);
      break;

    case G_TYPE_UINT:
      g_value_set_uint_by_string (value, string);
      break;

    case G_TYPE_LONG:
      g_value_set_long_by_string (value, string);
      break;

    case G_TYPE_ULONG:
      g_value_set_ulong_by_string (value, string);
      break;

    case G_TYPE_INT64:
      g_value_set_int64_by_string (value, string);
      break;

    case G_TYPE_UINT64:
      g_value_set_uint64_by_string (value, string);
      break;

    case G_TYPE_ENUM:
      g_value_set_enum_by_string (value, string);
      break;

    case G_TYPE_FLAGS:
      g_value_set_flags_by_string (value, string);
      break;

    case G_TYPE_FLOAT:
      g_value_set_float_by_string (value, string);
      break;

    case G_TYPE_DOUBLE:
      g_value_set_double_by_string (value, string);
      break;

    case G_TYPE_STRING:
      g_value_set_string (value, string);
      break;

    case G_TYPE_POINTER:
    case G_TYPE_BOXED:
    case G_TYPE_PARAM:
    case G_TYPE_OBJECT:
      /* XXX */
      break;

    default:
      /* XXX */
      break;
    }
}

void
g_value_set_by_static_string (GValue      *value,
			      const gchar *string)
{
  g_return_if_fail (value != NULL);
  g_return_if_fail (G_IS_VALUE (value));


  if (G_VALUE_HOLDS (value, G_TYPE_STRING))
    {
      g_value_set_static_string (value, string);
    }
  else
    {
      g_value_set_by_string (value, string);
    }
}
