/*
   Copyright (c) 2004, 2005 by AOSASA Shigeru and Red Hat, Inc.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:

   - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

   - Redistributions in binary form must reproduce the above
   copyright notice, this list of conditions and the following
   disclaimer in the documentation and/or other materials provided
   with the distribution.  

   - Neither the name of the AOSASA Shigeru, Red Hat, Inc
   nor the names of its contributors may be used to endorse or
   promote products derived from this software without specific
   prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
   COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
   INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
   (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
   SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
   STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
   OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*
   This software development was supported by
   Information-technorogy Promotion Agency, Japan (IPA).
*/

#include "gnode-utils.h"

void
g_node_remove_link (GNode *node)
{
  register GNode *parent;

  g_return_if_fail (node != NULL);


  parent = g_node_parent (node);
  g_node_unparent (node);
  g_node_forall_children (node,
			  (GNodeForallFunc)g_node_reparent,
			  (gpointer)parent);
  node->children = NULL;
}

void
g_node_unparent (GNode *node)
{
  g_return_if_fail (node != NULL);


  g_node_unlink (node);
}

void
g_node_reparent (GNode *node,
		 GNode *new_parent)
{
  g_return_if_fail (node != NULL);
  g_return_if_fail (new_parent != NULL);


  if (g_node_parent (node) != new_parent)
    {
      g_node_unparent (node);
      g_node_append (new_parent, node);
    }
}

void
g_node_forall_subtree (GNode           *top,
		       GNodeForallFunc  func,
		       gpointer         user_data)
{
  g_return_if_fail (top != NULL);
  g_return_if_fail (func != NULL);


  g_node_forall_descendants (top, func, user_data);
  (* func) (top, user_data);
}

void
g_node_forall_subtree_data (GNode    *top,
			    GFunc     func,
			    gpointer  user_data)
{
  g_return_if_fail (top != NULL);
  g_return_if_fail (func != NULL);


  g_node_forall_descendants_data (top, func, user_data);
  (* func) (top->data, user_data);
}

void
g_node_forall_siblings (GNode           *node,
			GNodeForallFunc  func,
			gpointer         user_data)
{
  register GNode *tmp;
  register GNode *sibling;

  g_return_if_fail (node != NULL);
  g_return_if_fail (func != NULL);


  tmp = g_node_prev_sibling (node);
  while (tmp != NULL)
    {
      sibling = tmp;
      tmp = g_node_prev_sibling (tmp);
      
      (* func) (sibling, user_data);
    }

  tmp = g_node_next_sibling (node);
  while (tmp != NULL)
    {
      sibling = tmp;
      tmp = g_node_next_sibling (tmp);

      (* func) (sibling, user_data);
    }
}

void
g_node_forall_siblings_data (GNode    *node,
			     GFunc     func,
			     gpointer  user_data)
{
  register GNode *tmp;
  register GNode *sibling;

  g_return_if_fail (node != NULL);
  g_return_if_fail (func != NULL);


  tmp = g_node_prev_sibling (node);
  while (tmp != NULL)
    {
      sibling = tmp;
      tmp = g_node_prev_sibling (tmp);
      
      (* func) (sibling->data, user_data);
    }

  tmp = g_node_next_sibling (node);
  while (tmp != NULL)
    {
      sibling = tmp;
      tmp = g_node_next_sibling (tmp);

      (* func) (sibling->data, user_data);
    }
}

void
g_node_forall_children (GNode           *node,
			GNodeForallFunc  func,
			gpointer         user_data)
{
  register GNode *tmp;
  register GNode *child;

  g_return_if_fail (node != NULL);
  g_return_if_fail (func != NULL);


  tmp = g_node_first_child (node);
  while (tmp != NULL)
    {
      child = tmp;
      tmp = g_node_next_sibling (tmp);

      (* func) (child, user_data);
    }
}

void
g_node_forall_children_data (GNode    *node,
			     GFunc     func,
			     gpointer  user_data)
{
  register GNode *tmp;
  register GNode *child;

  g_return_if_fail (node != NULL);
  g_return_if_fail (func != NULL);


  tmp = g_node_first_child (node);
  while (tmp != NULL)
    {
      child = tmp;
      tmp = g_node_next_sibling (tmp);

      (* func) (child->data, user_data);
    }
}

void
g_node_forall_ancestors (GNode           *node,
			 GNodeForallFunc  func,
			 gpointer         user_data)
{
  register GNode *tmp;
  register GNode *ancestor;

  g_return_if_fail (node != NULL);
  g_return_if_fail (func != NULL);


  tmp = g_node_parent (node);
  while (tmp != NULL)
    {
      ancestor = tmp;
      tmp = g_node_parent (tmp);

      (* func) (ancestor, user_data);
    }
}

void
g_node_forall_ancestors_data (GNode    *node,
			      GFunc     func,
			      gpointer  user_data)
{
  register GNode *tmp;
  register GNode *ancestor;

  g_return_if_fail (node != NULL);
  g_return_if_fail (func != NULL);


  tmp = g_node_parent (node);
  while (tmp != NULL)
    {
      ancestor = tmp;
      tmp = g_node_parent (tmp);

      (* func) (ancestor->data, user_data);
    }
}

void
g_node_forall_descendants (GNode           *node,
			   GNodeForallFunc  func,
			   gpointer         user_data)
{
  register GNode *tmp;
  register GNode *descendant;

  g_return_if_fail (node != NULL);
  g_return_if_fail (func != NULL);


  tmp = g_node_first_child (node);
  while (tmp != NULL)
    {
      descendant = tmp;
      tmp = g_node_next_sibling (tmp);
      
      if (!G_NODE_IS_LEAF (descendant))
	g_node_forall_descendants (descendant, func, user_data);

      (* func) (descendant, user_data);
    }
}

void
g_node_forall_descendants_data (GNode    *node,
				GFunc     func,
				gpointer  user_data)
{
  register GNode *tmp;
  register GNode *descendant;

  g_return_if_fail (node != NULL);
  g_return_if_fail (func != NULL);


  tmp = g_node_first_child (node);
  while (tmp != NULL)
    {
      descendant = tmp;
      tmp = g_node_next_sibling (tmp);
      
      if (!G_NODE_IS_LEAF (descendant))
	g_node_forall_descendants_data (descendant, func, user_data);

      (* func) (descendant->data, user_data);
    }
}

void
g_node_forall_depth (GNode           *node,
		     guint            depth,
		     GNodeForallFunc  func,
		     gpointer         user_data)
{
  register GNode *tmp;
  register GNode *child;

  g_return_if_fail (node != NULL);
  g_return_if_fail (func != NULL);
  g_return_if_fail (depth > 0);

  
  if (depth == 1)
    {
      (* func) (node, user_data);
    }
  else if (depth == 2)
    {
      g_node_forall_children (node, func, user_data);
    }
  else if (depth > 2)
    {
      tmp = g_node_first_child (node);
      while (tmp != NULL)
	{
	  child = tmp;
	  tmp = g_node_next_sibling (tmp);
	  
	  g_node_forall_depth (child, depth - 1, func, user_data);
	}
    }
}

void
g_node_forall_depth_data (GNode    *node,
			  guint     depth,
			  GFunc     func,
			  gpointer  user_data)
{
  register GNode *tmp;
  register GNode *child;

  g_return_if_fail (node != NULL);
  g_return_if_fail (func != NULL);
  g_return_if_fail (depth > 0);


  if (depth == 1)
    {
      (* func) (node->data, user_data);
    }
  else if (depth == 2)
    {
      g_node_forall_children_data (node, func, user_data);
    }
  else if (depth > 2)
    {
      tmp = g_node_first_child (node);
      while (tmp != NULL)
	{
	  child = tmp;
	  tmp = g_node_next_sibling (tmp);
	  
	  g_node_forall_depth_data (child, depth - 1, func, user_data);
	}
    }
}
