/*
 
    Ejecter - Safely, easily remove external peripherals
    Copyright 2008-2011, Federico Pelloni <federico.pelloni@gmail.com>
 
    
    This file is part of Ejecter.
 
    Ejecter is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Ejecter is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with Ejecter.  If not, see <http://www.gnu.org/licenses/>.
   
*/ 

using Gtk, GLib, Gdu, AppIndicator, Notify;


namespace Ejecter {
    
    public Ejecter.App ejecter;

public class App: GLib.Object {

    private AppIndicator.Indicator indicator;
    private GLib.VolumeMonitor monitor;
    private GLib.HashTable<string, Ejecter.Device> devices;
    private GLib.SList<string> invalid_devices;
    private Gtk.Menu menu;
    public Notify.Notification notification;
    private Gdu.Pool pool;

    construct {
    
        // AppIndicator
        this.indicator = new AppIndicator.Indicator("ejecter", "ejecter", AppIndicator.Category.HARDWARE);
        this.indicator.set_status(AppIndicator.Status.PASSIVE);
        
        // Main menu
        this.menu = new Gtk.Menu();
        
        Gtk.MenuItem header = new Gtk.MenuItem.with_label(_("Eject removable media"));
        header.set_sensitive(false);
        this.menu.append(header);
        header.show();
        
        Gtk.SeparatorMenuItem sep = new Gtk.SeparatorMenuItem();
        this.menu.append(sep);
        sep.show();
        
        this.menu.show();
        
        // Device map
        this.devices = new GLib.HashTable<string , Ejecter.Device>(GLib.str_hash, GLib.str_equal);        
        this.invalid_devices = new GLib.SList<string>();
        
        // GIO monitor
        this.pool = new Gdu.Pool();
        this.monitor = GLib.VolumeMonitor.get();
        this.load_devices();
        
        // Watchers
        this.monitor.volume_added.connect((monitor, volume) => { this.manage_volume(volume); });
        this.monitor.mount_added.connect((monitor, mount) => { this.manage_mount(mount); });
        
        this.indicator.set_menu(this.menu);
        
        // Notification
        if (!Notify.is_initted()) Notify.init(Config.PACKAGE_NAME);
        this.notification = new Notify.Notification(" ", "", "media-eject", null);
        this.notification.set_category("device");
        this.notification.set_urgency(Notify.Urgency.LOW);
        
    }
    
    
    
    public void notify_device (Ejecter.Device device) {
    
        this.notification.update(_("%s can be removed").printf(device.name),
                                 _("It's now possible to safely remove the device."), 
                                 "media-eject");
        try { this.notification.show(); }
        catch (GLib.Error error) { report_error(error); };
    
    }
    
    
    private void load_devices () {

        foreach (GLib.Volume v in (GLib.List<GLib.Volume>) this.monitor.get_volumes()) {
            
            debug("");
            
            GLib.Drive d = v.get_drive();
            
            this.manage_drive(d);
            this.manage_volume(v);
            
            GLib.Mount m = v.get_mount();
            if (m != null)
                this.manage_mount(m);

        }
        
        this.check_icon();

    }
    
    
    
    
    private Ejecter.Device? manage_drive (GLib.Drive? drive) {
    
        debug("");
        debug("NEW DEVICE");
        
        if (drive == null) { debug("Drive passed was null, skipping"); return null; };
    
        var id = drive.get_identifier(conf.device_identifier);
        if (id == null || id == "") { debug("No id"); return null; };
        if (this.invalid_devices.index(id) >= 0) { debug("Skipped drive (in invalid list)"); return null; };
        if (this.devices.lookup(id) != null) { debug("Skipped drive (already in list)"); return null; };
        debug("Drive id: " + id);
        
        Gdu.Device gdu_dev = this.pool.get_by_device_file(id);
        
        if (gdu_dev.is_system_internal() && !conf.show_internal) {
            debug("Device is internal: skip");
            this.invalid_devices.append(id);
            return null;
        }
        
        Ejecter.Device d = new Ejecter.Device(drive, gdu_dev);
        this.devices.insert(id, d);
        d.attach(this.menu);
        this.indicator.set_menu(this.menu);

        d.removed.connect(d => { 
                    this.devices.remove(d.drive.get_identifier(conf.device_identifier)); 
                    this.indicator.set_menu(this.menu);
                    this.check_icon();
                    try { this.notification.close(); }
                    catch (GLib.Error error) { report_error(error); };
                });
        d.unmounted.connect(d => { this.notify_device(d); });

        this.check_icon();

        return null;
        
    }
    
    
    private void manage_volume (GLib.Volume? volume) {
    
        debug("");                    
        debug("NEW VOLUME");
    
        if (volume == null) { debug("Volume was null, skipping"); return; };
    
        GLib.Drive drive = volume.get_drive();
        if (drive == null) { debug("Volume did not have a drive, skipping"); return; };
        
        string id = drive.get_identifier(conf.device_identifier);
        if (id == null || this.invalid_devices.index(id) >= 0) return;
        
        debug("Drive id: " + id);
        debug("Volume name: " + volume.get_name());
        
        Ejecter.Device dev = this.devices.lookup(id);
        
        if (dev == null) { 
            dev = this.manage_drive(drive);
            if (dev == null) { debug("No device - skipping"); return; };
        };
        
        dev.add_volume(volume);
        this.check_icon();

    }
    
    
    private void manage_mount (GLib.Mount? mount) {
    
        debug("");
        debug("NEW MOUNT");

        if (mount == null) { debug("Mount was null, skipping"); return; };

        GLib.Drive drive = mount.get_drive();
        
        if (drive == null) { debug("Mount did not have a drive, skipping"); return; };
        
        string id = drive.get_identifier(conf.device_identifier);
        if (id == null) return;
        
        debug("Mount id: " + id);
        debug("Mount name: " + mount.get_name());
        
        Ejecter.Device dev = this.devices.lookup(id);
        
        if (dev == null) return;
        
        dev.add_mount(mount);
        this.check_icon();

    }
   
   
    private void check_icon () {
        
        uint num = this.devices.size();
        
        if (num <= 0 && this.menu.visible) this.menu.popdown();

        if (num > 0) {
            this.indicator.set_status(AppIndicator.Status.ACTIVE);
        } else {
            try { if (this.notification != null) this.notification.close(); }
            catch (GLib.Error error) { report_error(error); };
            this.indicator.set_status(AppIndicator.Status.PASSIVE);
        }

    }
 
    
    
    static int main (string[] args) {
    
   		Intl.bindtextdomain(Config.GETTEXT_PACKAGE, Config.LOCALE_DIR);
		Intl.textdomain(Config.GETTEXT_PACKAGE);

        Gtk.init(ref args);
        
        GLib.Environment.set_application_name(Config.PACKAGE_NAME);
        
        // Options
        GLib.OptionEntry[] optentries = {
            GLib.OptionEntry () { long_name = "show-internal",
                                  short_name = 'i', 
                                  arg = GLib.OptionArg.NONE,
                                  description = "Also show internal partitions",
                                  arg_data = &conf.show_internal
                                }
        };
        
        var optcontext = new GLib.OptionContext("- "+_("Eject removable media"));
        optcontext.add_main_entries(optentries, Config.GETTEXT_PACKAGE);
        optcontext.set_help_enabled(true);
        optcontext.set_description(_("Easily and safely remove external media without losing data.\n\n"+
                                     "Click on the eject button next to the device you want to remove to unmount"+
                                     " it: when it has become greyed out you can safely plug it out from your "+
                                     "computer."));
        
        try {
            if (!optcontext.parse(ref args)) {
                debug("Error parsing options");
            }
        } catch (GLib.Error e) {
            report_error(e);
        }

        var ejecter = new Ejecter.App();
        Gtk.main();
        return 0;
    
    }

}

}

