(use files)

(define-syntax test
  (syntax-rules ()
    ((_ r x) (let ((y x)) (print y) (assert (equal? r y))))))

(test "/" (pathname-directory "/"))
(test "/" (pathname-directory "/abc"))
(test "abc" (pathname-directory "abc/"))
(test "abc" (pathname-directory "abc/def"))
(test "abc" (pathname-directory "abc/def.ghi"))
(test "abc" (pathname-directory "abc/.def.ghi"))
(test "abc" (pathname-directory "abc/.ghi"))
(test "/abc" (pathname-directory "/abc/"))
(test "/abc" (pathname-directory "/abc/def"))
(test "/abc" (pathname-directory "/abc/def.ghi"))
(test "/abc" (pathname-directory "/abc/.def.ghi"))
(test "/abc" (pathname-directory "/abc/.ghi"))
(test "q/abc" (pathname-directory "q/abc/"))
(test "q/abc" (pathname-directory "q/abc/def"))
(test "q/abc" (pathname-directory "q/abc/def.ghi"))
(test "q/abc" (pathname-directory "q/abc/.def.ghi"))
(test "q/abc" (pathname-directory "q/abc/.ghi"))

(test "./" (normalize-pathname "" 'unix))
(test ".\\" (normalize-pathname "" 'windows))
(test "\\..\\" (normalize-pathname "/../" 'windows))
(test "\\." (normalize-pathname "/abc/../." 'windows))
(test "/." (normalize-pathname "/" 'unix))
(test "/." (normalize-pathname "/./" 'unix))
(test "/." (normalize-pathname "/." 'unix))
(test "./" (normalize-pathname "./" 'unix))
(test "a" (normalize-pathname "a"))
(test "a/" (normalize-pathname "a/" 'unix))
(test "a/b" (normalize-pathname "a/b" 'unix))
(test "a/b" (normalize-pathname "a\\b" 'unix))
(test "a\\b" (normalize-pathname "a\\b" 'windows))
(test "a\\b" (normalize-pathname "a/b" 'windows))
(test "a/b/" (normalize-pathname "a/b/" 'unix))
(test "a/b/" (normalize-pathname "a/b//" 'unix))
(test "a/b" (normalize-pathname "a//b" 'unix))
(test "/a/b" (normalize-pathname "/a//b" 'unix))
(test "/a/b" (normalize-pathname "///a//b" 'unix))
(test "c:a\\b" (normalize-pathname "c:a/./b" 'windows))
(test "c:/a/b" (normalize-pathname "c:/a/./b" 'unix))
(test "c:a\\b" (normalize-pathname "c:a/./b" 'windows))
(test "c:b" (normalize-pathname "c:a/../b" 'windows))
(test "c:\\b" (normalize-pathname "c:\\a\\..\\b" 'windows))
(test "a/b" (normalize-pathname "a/./././b" 'unix))
(test "a/b" (normalize-pathname "a/b/c/d/../.." 'unix))
(test "a/b/" (normalize-pathname "a/b/c/d/../../" 'unix))
(test "../../foo" (normalize-pathname "../../foo" 'unix))
(test "c:\\." (normalize-pathname "c:\\" 'windows))

(define home (get-environment-variable "HOME"))

(when home
  (test (string-append home "/foo") (normalize-pathname "~/foo" 'unix))
  (test "c:~/foo" (normalize-pathname "c:~/foo" 'unix))
  (test (string-append home "\\foo") (normalize-pathname "c:~\\foo" 'windows)))

(assert (directory-null? "/.//"))
(assert (directory-null? ""))
(assert (not (directory-null? "//foo//")))

(test '(#f "/" (".")) (receive (decompose-directory "/.//")))
(test '(#f "\\" (".")) (receive (decompose-directory (normalize-pathname "/.//" 'windows))))
(test '(#f "/" #f) (receive (decompose-directory "///\\///")))
(test '(#f "/" ("foo")) (receive (decompose-directory "//foo//")))
(test '(#f "/" ("foo" "bar")) (receive (decompose-directory "//foo//bar")))
(test '(#f #f (" " "foo" "bar")) (receive (decompose-directory " //foo//bar")))
(test '(#f #f ("foo" "bar")) (receive (decompose-directory "foo//bar/")))

(test '(#f #f #f) (receive (decompose-pathname "")))
(test '("/" #f #f) (receive (decompose-pathname "/")))
(test '("\\" #f #f) (receive (decompose-pathname "\\")))
(test '("/" "a" #f) (receive (decompose-pathname "/a")))
(test '("\\" "a" #f) (receive (decompose-pathname "\\a")))
(test '("/" #f #f) (receive (decompose-pathname "///")))
(test '("\\" #f #f) (receive (decompose-pathname "\\\\\\")))
(test '("/" "a" #f) (receive (decompose-pathname "///a")))
(test '("\\" "a" #f) (receive (decompose-pathname "\\\\\\a")))
(test '("/a" "b" #f) (receive (decompose-pathname "/a/b")))
(test '("\\a" "b" #f) (receive (decompose-pathname "\\a\\b")))
(test '("/a" "b" "c") (receive (decompose-pathname "/a/b.c")))
(test '("\\a" "b" "c") (receive (decompose-pathname "\\a\\b.c")))
(test '("." "a" #f) (receive (decompose-pathname "./a")))
(test '("." "a" #f) (receive (decompose-pathname ".\\a")))
(test '("." "a" "b") (receive (decompose-pathname "./a.b")))
(test '("." "a" "b") (receive (decompose-pathname ".\\a.b")))
(test '("./a" "b" #f) (receive (decompose-pathname "./a/b")))
(test '(".\\a" "b" #f) (receive (decompose-pathname ".\\a\\b")))
(test '(#f "a" #f) (receive (decompose-pathname "a")))
(test '(#f "a." #f) (receive (decompose-pathname "a.")))
(test '(#f ".a" #f) (receive (decompose-pathname ".a")))
(test '("a" "b" #f) (receive (decompose-pathname "a/b")))
(test '("a" "b" #f) (receive (decompose-pathname "a\\b")))
(test '("a" "b" #f) (receive (decompose-pathname "a///b")))
(test '("a" "b" #f) (receive (decompose-pathname "a\\\\\\b")))
(test '("a/b" "c" #f) (receive (decompose-pathname "a/b/c")))
(test '("a\\b" "c" #f) (receive (decompose-pathname "a\\b\\c")))
(test '("a/b/c" #f #f) (receive (decompose-pathname "a/b/c/")))
(test '("a\\b\\c" #f #f) (receive (decompose-pathname "a\\b\\c\\")))
(test '("a/b/c" #f #f) (receive (decompose-pathname "a/b/c///")))
(test '("a\\b\\c" #f #f) (receive (decompose-pathname "a\\b\\c\\\\\\")))
(test '(#f "a" "b") (receive (decompose-pathname "a.b")))
(test '("a.b" #f #f) (receive (decompose-pathname "a.b/")))
(test '("a.b" #f #f) (receive (decompose-pathname "a.b\\")))
(test '(#f "a.b" "c") (receive (decompose-pathname "a.b.c")))
(test '(#f "a." "b") (receive (decompose-pathname "a..b")))
(test '(#f "a.." "b") (receive (decompose-pathname "a...b")))
(test '("a." ".b" #f) (receive (decompose-pathname "a./.b")))
(test '("a." ".b" #f) (receive (decompose-pathname "a.\\.b")))

(test "x/y/z.q" (make-pathname "x/y" "z" "q"))
(test "x/y/z.q" (make-pathname "x/y" "z.q"))
(test "x/y/z.q" (make-pathname "x/y/" "z.q"))
(test "x/y/z.q" (make-pathname "x/y/" "z.q"))
(test "x/y/z.q" (make-pathname "x/y\\" "z.q"))
(test "x//y/z.q" (make-pathname "x//y/" "z.q"))
(test "x\\y/z.q" (make-pathname "x\\y" "z.q"))
(test 'error (handle-exceptions _ 'error (make-pathname '(#f) "foo")))
(test "/x/y/z" (make-pathname #f "/x/y/z"))
(test "/x/y/z" (make-pathname "/" "x/y/z"))
(test "/x/y/z" (make-pathname "/x" "/y/z"))
(test "/x/y/z" (make-pathname '("/") "x/y/z"))
(test "/x/y/z" (make-pathname '("/" "x") "y/z"))
(test "/x/y/z" (make-pathname '("/x" "y") "z"))
(test "/x/y/z/" (make-pathname '("/x" "y" "z") #f))
