/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef MESH_DATA_MODEL_H
#define MESH_DATA_MODEL_H

#include <QAbstractTableModel>
#include <QSortFilterProxyModel>

namespace camitk {

class MeshComponent;

/**
 * Qt model for mesh data.
 * This class use the Qt model/view design
 */
class MeshDataModel : public QAbstractTableModel {

    Q_OBJECT

public :

    /// @enum FieldType Data fields can be applied to one of this
    enum FieldType {
        POINTS = 1,  ///< data are attached to point
        CELLS = 2,   ///< data are attached to cells
        MESH = 4     ///< data are attached to the whole mesh
    };

    /// @enum DataType Data fields can have different dimensions
    enum DataType {
        SCALARS = 1,  ///< 1D (scalar value)
        VECTORS = 2,  ///< 3D (3D vector)
        TENSORS = 4,  ///< 9D (3x3 matrix)
        OTHERS = 8    ///< other dimensions (warning: nothing special are managed by this class, no specific interaction)
    };

    /**
     * @brief Constructor
     */
    MeshDataModel ( MeshComponent* meshComp );

    /**
     * @brief Number of data arrays
     */
    int rowCount ( const QModelIndex &parent = QModelIndex() ) const ;

    /**
     * @brief Number of data arrays columns
     */
    int columnCount ( const QModelIndex &parent = QModelIndex() ) const;

    /**
     * @brief Model data, called when the view is refreshing visualization
     */
    QVariant data ( const QModelIndex &index, int role = Qt::DisplayRole ) const;

    /**
     * @brief Header data
     */
    QVariant headerData ( int section, Qt::Orientation orientation, int role = Qt::DisplayRole ) const;

    /**
     * @brief Edit data, called when the data are modified by the user (view)
     */
    bool setData ( const QModelIndex &index, const QVariant &value, int role );

    /// Returns if a given model index is editable, checkable....
    /// @see QAbstractTableModel
    Qt::ItemFlags flags ( const QModelIndex &index ) const;

    /// static method that returns the FieldType enum as a QString
    static const QMap< int, QString >& getFieldNames();

    /// static method that returns the DataType enum as a QString
    static const QMap< int, QString >& getDataNames();

    /**
     * @brief Refresh the model
     */
    void refresh();

private :

    /// The component where the data are stored
    MeshComponent* meshComponent;

    /// initialize FieldType QString map
    static QMap< int, QString > initFieldNames();

    /// initialize DataType QString map
    static QMap< int, QString > initDataNames();

    /// determine the current field type of the item depending on the row index, compute the index of the data in its field category
    FieldType getFieldTypeOfRow(const int, int *dataIndex) const;

};

class MeshDataFilterModel : public QSortFilterProxyModel {

    Q_OBJECT

public :

    MeshDataFilterModel(int fieldFilter = MeshDataModel::POINTS | MeshDataModel::CELLS | MeshDataModel::MESH,
                        int dataFilter = MeshDataModel::SCALARS | MeshDataModel::VECTORS | MeshDataModel::TENSORS | MeshDataModel::OTHERS,
                        QObject* parent = NULL);

    void setFieldTypeFilter(int fieldFilter);

    void setDataTypeFilter(int dataFilter);

protected :

    bool filterAcceptsRow(int sourceRow, const QModelIndex &sourceParent) const;

private :

    int fieldTypeFilter;

    int dataTypeFilter;

};

}

#endif
