#! /usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Copyright 2021, Nils Hilbricht, Germany ( https://www.hilbricht.net )

This file is part of the Laborejo Software Suite ( https://www.laborejo.org ),

This application is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import logging; logger = logging.getLogger(__name__); logger.info("import")


#Standard Library
import pathlib
import os

#Third Party
from PyQt5 import QtCore, QtWidgets

#Engine
import engine.api as api
from engine.config import METADATA  #includes METADATA only. No other environmental setup is executed.

#QtGui
from .designer.settings import Ui_Dialog

class SettingsDialog(QtWidgets.QDialog):

    def __init__(self, mainWindow):
        super().__init__()
        self.ui =  Ui_Dialog()
        self.ui.setupUi(self)
        self.mainWindow = mainWindow

        self.success = False

        logger.info("Init settings dialog")

        settings = QtCore.QSettings("LaborejoSoftwareSuite", METADATA["shortName"])
        if settings.contains("launcherBlacklistPlainTextEdit"):
            self.ui.launcherBlacklistPlainTextEdit.setPlainText(settings.value("launcherBlacklistPlainTextEdit", type=str))
        else:
            self.ui.launcherBlacklistPlainTextEdit.setPlainText("")

        if settings.contains("launcherWhitelistPlainTextEdit"):
            self.ui.launcherWhitelistPlainTextEdit.setPlainText(settings.value("launcherWhitelistPlainTextEdit", type=str))
        else:
            self.ui.launcherWhitelistPlainTextEdit.setPlainText("")

        if settings.contains("programPathsPlainTextEdit"):
            self.ui.programPathsPlainTextEdit.setPlainText(settings.value("programPathsPlainTextEdit", type=str))
        else:
            self.ui.programPathsPlainTextEdit.setPlainText("")

        #self.ui.name.textEdited.connect(self.check) #not called when text is changed programatically

        self.ui.buttonBox.accepted.connect(self.process)
        self.ui.buttonBox.rejected.connect(self.reject)

        self.setWindowFlag(QtCore.Qt.Popup, True)
        self.setModal(True)
        self.setFocus(True)
        logger.info("Show settings dialog")
        self.exec_()

    @staticmethod
    def loadFromSettingsAndSendToEngine():
        """Called on program start and in self.process, which has a bit overhead because
        it is saving to file and then reloading from file (qsettings)"""
        settings = QtCore.QSettings("LaborejoSoftwareSuite", METADATA["shortName"])
        if settings.contains("launcherBlacklistPlainTextEdit"):
            bl = settings.value("launcherBlacklistPlainTextEdit", type=str)
        else:
            bl = None

        if settings.contains("launcherWhitelistPlainTextEdit"):
            wl = settings.value("launcherWhitelistPlainTextEdit", type=str)
        else:
            wl = None

        if settings.contains("programPathsPlainTextEdit"):
            pth = settings.value("programPathsPlainTextEdit", type=str)
        else:
            pth = None

        blacklist = bl.split("\n") if bl else []
        whitelist = wl.split("\n") if wl else []

        api.systemProgramsSetBlacklist(blacklist)
        api.systemProgramsSetWhitelist(whitelist)

        #Depends on SettingsDialog: More executable paths for the engine. We do this in mainwindow because it has access to the qsettings safe file and is started before engine, program-database or nsmd.
        additionalExecutablePaths = pth.split("\n") if pth else []
        if additionalExecutablePaths:
            os.environ["PATH"] = os.pathsep.join(additionalExecutablePaths) + os.pathsep + os.environ["PATH"]
        logger.info(f"Binary search paths: {os.environ['PATH']}")


    def process(self):
        settings = QtCore.QSettings("LaborejoSoftwareSuite", METADATA["shortName"])

        settings.setValue("launcherBlacklistPlainTextEdit", self.ui.launcherBlacklistPlainTextEdit.toPlainText())
        settings.setValue("launcherWhitelistPlainTextEdit", self.ui.launcherWhitelistPlainTextEdit.toPlainText())
        settings.setValue("programPathsPlainTextEdit", self.ui.programPathsPlainTextEdit.toPlainText())

        SettingsDialog.loadFromSettingsAndSendToEngine()

        self.success = True
        self.done(True)
