/* -*- C++ -*-
   (c) 2008 Peter Rockai <me@mornfall.net>,
   (c) 2008 Yuriy Kozlov <yuriy.kozlov@gmail.com> */

#include <adept/packagedata.h>
#include <adept/util.h>

#include <QtGui/QWidget>
#include <QtGui/QCheckBox>

#include <QtCore/QRegExp>
#include <QtCore/QDebug>

#include <ept/core/apt.h>
#include <ept/core/apt/action.h>

#ifndef ADEPTPACKAGEINFO_H
#define ADEPTPACKAGEINFO_H

namespace adept {

using namespace ept::core;

/**
 * @class PackageInfo
 * Abstract class with common fields and methods for displaying
 * package information.  Provides a checkboxes and methods for
 * changing the package state.
 */
class PackageInfo : public QWidget
{
    Q_OBJECT
protected:
    ept::Token t;
    ept::Token desktop;
    PackageData &d;

    QCheckBox *m_act, *m_remove;
    bool block;
    
public:
    PackageInfo( PackageData &_d, QWidget *parent = 0 )
        : QWidget( parent ), d( _d ), m_act( 0 ), m_remove( 0 ), block( false )
    {
    }
    
    PackageInfo( PackageData &_d, ept::Token _t, QWidget *parent = 0 )
        : QWidget( parent ), t( _t ), d( _d ), m_act( 0 ),
          m_remove( 0 ), block( false )
    {
        if ( t.isDesktop() ) {
            desktop = t;
            t = d.desk.get< desktop::Package >( t );
        }
    }

    ept::Token token() { return t; }

    ~PackageInfo() {};
    
public Q_SLOTS:
    virtual void refresh() = 0;

    void removeStateChanged( int s ) {
        if ( block )
            return;
        block = true;
        if ( s == Qt::Unchecked ) {
            d.actl.add( package::Action( t, package::Action::Keep ) );
        } else {
            m_act->setChecked( false );
            d.actl.add( package::Action( t, package::Action::Remove ) );
        }
        d.pkgs.revertStates();
        d.actl.replay( d.pkgs );
        block = false;
        changed();
    }

    void actStateChanged( int s ) {
        if ( block )
            return;
        block = true;
        if ( s == Qt::Unchecked ) {
            d.actl.add( package::Action( t, package::Action::Keep ) );
        } else {
            PackageState st = d.pkgs.get< package::State >( t );
            if (m_remove)
                m_remove->setChecked( false );
            if ( st.installed() && !st.hasNewVersion() )
                d.actl.add( package::Action( t, package::Action::Remove ) );
            else
                d.actl.add( package::Action( t, package::Action::Install ) );
        }
        d.pkgs.revertStates();
        d.actl.replay( d.pkgs );
        block = false;
        changed();
    }

    void refreshCheckboxes( QWidget *parent, PackageState s )
    {
        block = true;
        if ( !m_act ) {
            m_act = new QCheckBox( parent );
            connect( m_act, SIGNAL( stateChanged( int ) ), this, 
                     SLOT( actStateChanged( int ) ) );
        }

        if ( s.hasNewVersion() ) {
            m_act->setText( i18n( "Upgrade this package" ) );
            if ( !m_remove ) {
                m_remove = new QCheckBox( parent );
                m_remove->setText( i18n( "Remove this package" ) );
                connect( m_remove, SIGNAL( stateChanged( int ) ), this, 
                         SLOT( removeStateChanged( int ) ) );
            }
            m_act->setChecked( s.upgrade() );
            m_remove->setChecked( s.remove() );
        } else if ( s.installed() ) {
            delete m_remove;
            m_remove = 0;
            m_act->setText( i18n( "Remove this package" ) );
            m_act->setChecked( s.remove() );
        } else {
            delete m_remove;
            m_remove = 0;
            m_act->setText( i18n( "Install this package" ) );
            m_act->setChecked( s.install() );
        }
        block = false;
    }
    
Q_SIGNALS:
    void changed();

};

}

#endif
