/* -*- C++ -*-
   (c) 2008 Peter Rockai <me@mornfall.net>
   (c) 2008 Yuriy Kozlov <yuriy.kozlov@gmail.com> */

#include <adept/packagedetails.h>

#include <QtCore/QFile>

#include <KDirLister>

using namespace adept;

PackageDetails::PackageDetails( PackageData &_d, QWidget *parent )
    : PackageInfo( _d, parent )
{
    m_layout = new QVBoxLayout( this );

    setupInfo();
    setupTabWidget();

    m_needExpensiveRefresh = false;
    
    connect( this, SIGNAL( changed() ), this, SLOT( refresh() ) );
}

void PackageDetails::loadFileList() {
    PackageState s = d.pkgs.get< package::State >( t );
    if ( !s.installed() ) {
        return;
    }
    QString flfilename = "/var/lib/dpkg/info/" + u8(t.package()) + ".list";
    QFile flfile(flfilename);
    if (!flfile.open(QIODevice::ReadOnly | QIODevice::Text)) {
        qDebug() << "Failed to open file list, error:" << flfile.error() << endl;
        return;
    }
    qint64 flFileSize = flfile.size();
    // If the list is small, don't bother showing progress
    bool needProgressBar = ( flFileSize > 1000 );

    qDebug() << "PackageDetails::loadFileListWorker() entering loop" << endl;

    if ( needProgressBar ) {
        m_fileLoadingProgress->reset();
        // Approximate progress by approximating the number of lines to be read in.
        // Judging by some files in /var/lib/dpkg/info/*.list, 50-60 bytes/line
        // is reasonable.
        m_fileLoadingProgress->setMaximum(flFileSize / 53);
        m_fileLoadingProgress->show();
    }
    
    m_fileProxy->clear();
    
    QTextStream flInStream(&flfile);
    int i = 0;
    while (!flInStream.atEnd()) {
        QString line = flInStream.readLine();
        if ( line == "/." || line == "" )
            continue; // nasty evil thing go away
        KUrl url( "file:///" );
        url.addPath( line );
        qDebug() << url;
        m_fileProxy->addUrl( url );
        ++i;
        m_fileLoadingProgress->setValue(i);
    }
    flfile.close();
    
    m_fileProxy->apply();
    qDebug() << "PackageDetails::loadFileListWorker() leaving loop" << endl;
    m_fileList->expandToDepth( 2 );
    if ( needProgressBar ) {
        m_fileLoadingProgress->setValue(m_fileLoadingProgress->maximum());
        m_fileLoadingProgress->hide();
    }
}
    
void PackageDetails::refresh(){
    setUpdatesEnabled( false );
    m_needExpensiveRefresh = true;

    PackageState s = d.pkgs.get< package::State >( t );
    
    QString name = u8( t.package() ) + ": ",
            status = statusString( s ),
            comma = u8( " — " ),
            action = actionString( s ),
            shortdesc = u8( d.recs.get< record::ShortDescription >( t ) );
    QString nameStatusLabelStr = "<b>" + name + "</b>" + 
        "<font color=" + statusColor( s ).name() + ">" + status + "</font>";
    if ( !s.keep() ) {
        nameStatusLabelStr += comma
                + "<font color=" + actionColor( s ).name() + ">" + action + "</font>";
    }
    m_pkgNameLabel->setText(nameStatusLabelStr);
    m_shortDesc->setText(shortdesc);
            
    QString versionInfo;
    ept::Token candidate = d.pkgs.get< package::CandidateVersion >( t ),
                installed = d.pkgs.get< package::InstalledVersion >( t );
    if ( s.hasNewVersion() ) {
        versionInfo = i18n( "<nobr>Version<b>" )
                + u8( installed.version() )
                + i18n( "</b> is installed,</nobr> <nobr>version <b>" )
                + u8( candidate.version() )
                + i18n( "</b> is available.</nobr>" );
    } else if ( !s.installed() && d.pkgs.exists( candidate ) ) {
        versionInfo = i18n( "<nobr>Version <b>" )
                + u8( candidate.version() )
                + i18n( "</b> is available.</nobr>" );
    } else if ( d.pkgs.exists( installed ) ) {
        versionInfo = i18n( "<nobr>Version <b>" )
                + u8( installed.version() )
                + i18n( "</b> is installed.</nobr>" );
    } else {
        versionInfo = i18n( "<nobr>No versions available.</nobr>" );
    }

    QString maintainer = i18n( "Maintainer: <b>" )
                         + u8( d.recs.get< record::Maintainer >( t ) )
                         + i18n( "</b>" ),
               section = i18n( "Section: <b>" )
                         + u8( d.recs.get< record::Section >( t ) )
                         + i18n( "</b>" ),
         installedSize = i18n( "Installed Size: <b>" )
                         + QString::number(
                             d.recs.get< record::InstalledSize >( t ) ) + "kB"
                         + i18n( "</b>" ),
         sourcePackage = i18n( "Source Package: <i>" )
                         + u8( d.recs.get< record::SourcePackage >( t ) )
                         + i18n( "</i>" ),
          architecture = i18n( "Architecture: <b>" )
                         + u8( d.recs.get< record::Architecture >( t ) )
                         + i18n( "</b>" );

    m_desc->setText( formatLongDescription(
                            u8( d.recs.get< record::LongDescription >( t ) ) ) );
    m_versionInfo->setText( versionInfo );
    m_maintainer->setText( maintainer );
    m_section->setText( section );
    m_installedSize->setText( installedSize );
    m_sourcePackage->setText( sourcePackage );
    m_architecture->setText( architecture );

    refreshCheckboxes( m_infoBox, s );
    
    m_bottom->setTabEnabled( 3, s.installed() );
    setUpdatesEnabled( true );
    update();
}
    
#include "packagedetails.moc"
