/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
*/
package gov.nasa.worldwind.servers.wms;

import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.servers.wms.xml.BoundingBox;
import gov.nasa.worldwind.servers.wms.xml.EXGeographicBoundingBox;
import gov.nasa.worldwind.servers.wms.xml.Keyword;
import gov.nasa.worldwind.servers.wms.xml.KeywordList;
import gov.nasa.worldwind.servers.wms.xml.Layer;
import gov.nasa.worldwind.servers.wms.xml.ObjectFactory;
import java.io.ByteArrayOutputStream;
import java.util.Iterator;
import java.util.Properties;
import java.util.Vector;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;

/**
 * MapSource corresponds to the <code>&lt;MapSource&gt;</code> configuration elements in the
 * <code>WEB-INF/config.xml</code> configuration file. The XML form for a MapSource is:
 *
 * <pre>
 *     &lt;MapSource name=" layer name " title=" layer title "&gt;
 *         &lt;description keywords="..."&gt; layer description &lt;/description&gt;
 *         &lt;root-dir&gt; path to root of data directory for layer &lt;/root-dir&gt;
 *         &lt;class&gt;...name of class implementing MapGenerator to serve layer...&lt;/class&gt;
 *         &lt;!-- zero or more properties recognized by MapGenerator class
 *             &lt;property name="..." value="..." /&gt;
 *          --&gt;
 *     &lt;/MapSource&gt;
 * </pre>
 *
 * <p>
 * See the javadoc for the optional or required properties recognized by a specific MapGenerator
 * implementing class.
 * </p>
 * <p>Note that the name, title, keyword attributes, along with the description element, are
 * used in the WMS GetCapabilities response, and are thus potentially forward-facing text strings.
 * </p>
 *
 * @author brownrigg
 * @version $Id: MapSource.java 4899 2008-04-02 17:10:06Z rick $
 */
public class MapSource {

    public MapSource(MapSource parent, String name, String title, String rootDir, Class serviceClass, String description,
                     String keywords, Properties properties) {
        this.parent = parent;
        this.name = name.replace(",", "_");  // commas trip up "Layers=" clause in GetMap/Feature requests...
        this.title = title;
        this.rootDir = rootDir;
        this.serviceClass = serviceClass;
        this.description = description;
        this.keywords = keywords;
        this.properties = properties;
        this.nestedMapSources = new Vector<MapSource>(1);
    }

    public String getName() {
        return this.name;
    }
    
    public String getFullName() {
        String parentPrefix;
        if (parent == null) 
            parentPrefix = nestedNamesSep;
        else
            parentPrefix = parent.getFullName() + nestedNamesSep;
        String fullname = parentPrefix + ((isNamed()) ? getName() : getTitle());
        return fullname.replace(",", "_");    // commas trip up "Layers=" clause in GetMap/Feature requests...
    }
    
    public boolean isNamed() {
        return (name == null || name.equals("")) ? false : true;
    }

    public String getTitle() {
        return this.title;
    }

    public String getRootDir() {
        return this.rootDir;
    }

    public Class getServiceClass() {
        return this.serviceClass;
    }

    public String getDescription() {
        return this.description;
    }

    public String getKeywords() {
        return this.keywords;
    }

    public Properties getProperties() {
        return this.properties;
    }

    public MapGenerator getMapGenerator() throws Exception {
        if (this.mapGenerator == null) {
            this.mapGenerator = (MapGenerator) this.getServiceClass().newInstance();
        }

        return this.mapGenerator;
    }
    
    public void addChild(MapSource ms) {
        nestedMapSources.add(ms);
    }
    
    public Iterator<MapSource> getChildren() { 
        return nestedMapSources.iterator();
    }
    
    public String getCapsLayerXML() {
        String layerStr=null;
        try{
            Layer layer = this.getCapsLayerObject();
            
            JAXBContext jaxbContext = JAXBContext.newInstance(Layer.class);
            Marshaller marshaller = jaxbContext.createMarshaller();
            marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
            marshaller.setProperty(Marshaller.JAXB_FRAGMENT, true);

            ByteArrayOutputStream stream=new ByteArrayOutputStream();
            marshaller.marshal(layer, stream);
            layerStr = stream.toString();
        } catch(Exception ex) {
            SysLog.inst().error("Exception building layer info, MapSource=" + this.getName() + ": " + ex.getMessage());
            layerStr=null;
        }

        return layerStr;
    }
    
    private Layer getCapsLayerObject() throws Exception {
        ObjectFactory of = new ObjectFactory();
        Layer layer = of.createLayer();
        layer.setQueryable(false);
        if (this.isNamed()) {
            layer.setName(this.getFullName());
        }
        layer.setTitle(this.getTitle());
        layer.setAbstract(this.getDescription());

        String keywordsStr = this.getKeywords();
        if (keywordsStr != null) {
            KeywordList kwl = new KeywordList();
            //parse keywordsStr
            String[] keys = keywordsStr.split(";");
            for (int i = 0; i < keys.length; i++) {
                Keyword kw = new Keyword();
                kw.setValue(keys[i]);
                kwl.getKeyword().add(kw);
            }
            layer.setKeywordList(kwl);
        }
        
        // TODO: Support for hierarchical (named or unamed) Layers not fully implemented here!
        if (this.isNamed()) {
            //BoundingBox
            MapGenerator mapGen = this.getMapGenerator();
            if (mapGen != null) {
                Sector bnds = mapGen.getBBox();
                if (bnds != null) {
                    BoundingBox bbox = new BoundingBox();
                    bbox.setCRS(mapGen.getCRS()[0]); // TODO: ONLY ONE FOR NOW

                    bbox.setMaxx(bnds.getMaxLongitude().degrees);
                    bbox.setMinx(bnds.getMinLongitude().degrees);
                    bbox.setMaxy(bnds.getMaxLatitude().degrees);
                    bbox.setMiny(bnds.getMinLatitude().degrees);
                    layer.getBoundingBox().add(bbox);

                    EXGeographicBoundingBox exGeoBox = new EXGeographicBoundingBox();
                    exGeoBox.setEastBoundLongitude(bnds.getMaxLongitude().degrees);
                    exGeoBox.setWestBoundLongitude(bnds.getMinLongitude().degrees);
                    exGeoBox.setNorthBoundLatitude(bnds.getMaxLatitude().degrees);
                    exGeoBox.setSouthBoundLatitude(bnds.getMinLatitude().degrees);
                    layer.setEXGeographicBoundingBox(exGeoBox);
                }
                layer.getCRS().add(mapGen.getCRS()[0]);  // TODO: ONLY ONE FOR NOW
            }
        }
        
        // include Layers for any children...
        Iterator<MapSource> iter = this.getChildren();
        while (iter.hasNext()) {
            layer.getLayer().add(iter.next().getCapsLayerObject());
        }

        return layer;
    }

    private MapSource parent;
    private String name;
    private String title;
    private String rootDir;
    private Class serviceClass;
    private String description;
    private String keywords;
    private Properties properties;
    private MapGenerator mapGenerator;
    private Vector<MapSource> nestedMapSources;
    
    private static final String nestedNamesSep = "|";
}
