/* NumberOfCenterVertices.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.other;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.invariants.values.IntegerValue;

/**
 * Invariant computer which returns the number of center vertices, i.e. the number of vertices
 * with eccentricity equal to the radius.
 */
public class NumberOfCenterVertices extends AbstractInvariantComputer {
    
    public IntegerValue compute(GraphBundleView bundle) {
        int[] eccentricityList = bundle.eccentricityList();

        int n = eccentricityList.length;
        if (n <= 1)
                return new IntegerValue (n, this);
        if(eccentricityList[1]==Integer.MAX_VALUE)
            return IntegerValue.undefinedValue (this); //undefined for disconnected graphs
        int radius = Integer.MAX_VALUE;
        int count = 0;
        for (int eccentricity : eccentricityList) {
            if (eccentricity == radius)
                count++;
            else if (eccentricity < radius) {
                radius = eccentricity;
                count = 1;
            }
        }
        
        return new IntegerValue(count, this);
    }

    public String getInvariantId() {
        return "org.grinvin.invariants.NumberOfCenterVertices";
    }
    
}
