/* DefaultGraphIconFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.icons;

import javax.swing.Icon;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.graphs.GraphView;
import org.grinvin.graphs.render.DefaultRenderer;

/**
 * Factory which is to be used when a graph bundle does not explicitely
 * request an icon factory.
 */
public class DefaultGraphIconFactory implements GraphIconFactory {
    
    /**
     * Default constructor. Can be overridden, but should not
     * be called directly.
     * @see #getInstance
     */
    protected DefaultGraphIconFactory() { 
        // can be overridden but cannot be called directly
    }
    
    //
    private static final GraphIconFactory SINGLETON
            = new DefaultGraphIconFactory();
    
    /**
     * Return the sole instance of this class.
     */
    public static GraphIconFactory getInstance() {
        return SINGLETON;
    }
    
    /**
     * Determine whether a graph qualifies as 'small'. Tyically
     * small graphs use an icon that closely represents the graph
     * embedding while large graphs use a more general icon.<p>
     * This implementation checks whether the number of vertices
     * is smaller than the square of the size divide by the
     * vertex limit factor.
     * @param size Size of the icon
     * @see #getVertexLimitFactor
     */
    protected boolean isSmallGraph(GraphBundleView bundle, int size) {
        return bundle.getGraph().getNumberOfVertices()*getVertexLimitFactor()
        <= size*size;
    }
    
    protected boolean isHugeGraph(GraphBundleView bundle) {
        return bundle.getGraph().getNumberOfEdges() + bundle.getGraph().getNumberOfVertices() > 1000;
    }
    
    /**
     * Return the vertex limit factor used to determine whether a graph should
     * be classified as 'small' or 'large'.
     * @see #isSmallGraph
     */
    protected int getVertexLimitFactor() {
        return 300;
    }
    
    /**
     * Create an icon for a 'small' graph.
     */
    protected GraphIcon createIconForSmallGraph(GraphBundleView bundle, int size) {
        return new SmallGraphIcon(bundle, DefaultRenderer.getInstance(),
                size); // TODO: use CachedSmallGraphIcon?
    }
    
    /**
     * Return the background icon for the large graph icon.
     */
    protected Icon getBackgroundIcon(int size) {
        return SvgIconManager.getInstance().getIcon("/org/grinvin/icons/graph.svg", size);
    }
    
    /**
     * Create an icon for a 'large' graph.
     */
    protected GraphIcon createIconForLargeGraph(GraphBundleView bundle, int size) {
        return new LargeGraphIcon(bundle.getGraph(), getBackgroundIcon(size));
    }
    
    /**
     * Create an icon for a graph bundle whose graph is missing.
     */
    protected GraphIcon createIconForMissingGraph(GraphBundleView bundle, int size) {
        return new MissingGraphIcon(size);
    }
    
    //
    public GraphIcon createIcon(GraphBundleView bundle, int size) {
        GraphView graph = bundle.getGraph();
        if (graph == null) {
            return createIconForMissingGraph(bundle, size);
        } else if (isSmallGraph(bundle, size) && !isHugeGraph(bundle))
            return createIconForSmallGraph(bundle, size);
        else
            return createIconForLargeGraph(bundle, size);
    }
    
}
