/* ExpressionToFormula.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.expr;

import be.ugent.caagt.jmathtex.TeXFormula;

import org.grinvin.invariants.Invariant;

/**
 * Visitor which converts an expression to a formula.<p>
 * <b>Note:</b> This class is not thread safe.
 */
public class ExpressionToFormula implements Expression.Visitor {
    
    //
    private TeXFormula result = new TeXFormula();
    
    //
    @SuppressWarnings("PMD.SingularField")
    private ExpressionToFormula childVisitor;
    
    /** 
     * Get a visitor for visiting children of expressions.
     */
    private ExpressionToFormula getChildVisitor () {
        if (childVisitor == null)
            childVisitor = new ExpressionToFormula ();
        return childVisitor;
    }
    
    /**
     * Return the resulting formula.
     */
    public TeXFormula getFormula () {
        return this.result;
    }
    
    //
    private int strength;
    
    /**
     * Return the strength of the resulting formula. (Used to determine
     * whether parentheses are needed.)
     */
    public int getStrength () {
        return this.strength;
    }
    
    public void visitInvariant(Invariant invariant) {
        this.result = new TeXFormula(invariant.getName());
        this.strength = Integer.MAX_VALUE;
    }
    
    public void visitIntegerLiteral(IntegerLiteral literal) {
        this.result = new TeXFormula(literal.toString());
        this.strength = Integer.MAX_VALUE;
    }

    public void visitCompound(Compound compound) {
        Operator operator = compound.getOperator();
                ExpressionToFormula childVisitor = getChildVisitor ();
        int numberOfArguments = compound.getNumberOfChildren();
        this.strength = operator.getStrength();
        if (numberOfArguments == 1) {
            // unary operator
            
            compound.getChild(0).accept (childVisitor);
            TeXFormula formula = childVisitor.getFormula();
            if (this.strength >= childVisitor.getStrength())
                formula = new TeXFormula().addEmbraced(formula, '(', ')');
            this.result = new TeXFormula(operator.getSymbol()).add(formula);
            
        } else if (numberOfArguments == 2) {
            // binary operator (assumed left associative)
            
            compound.getChild(0).accept (childVisitor);
            TeXFormula formula1 = childVisitor.getFormula();
            if (this.strength > childVisitor.getStrength())
                formula1 = new TeXFormula().addEmbraced(formula1, '(', ')');
            
            compound.getChild(1).accept (childVisitor);
            TeXFormula formula2 = childVisitor.getFormula();
            if (this.strength >= childVisitor.getStrength())
                formula2 = new TeXFormula().addEmbraced(formula2, '(', ')');
            if(StandardOperator.DIV.equals(operator))
                this.result = new TeXFormula().addFraction(formula1, formula2, true);
            else if(StandardOperator.LE.equals(operator))
                this.result= new TeXFormula(formula1).addSymbol("leq").add(formula2);
            else
                this.result= new TeXFormula(formula1).add(operator.getSymbol()).add(formula2);
            
        } else {
            assert false;
        }
    }
    
}
