/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */

using GLib;

namespace Unity.MusicLens {
 	
  public class BansheeScopeProxy : SimpleScope
  { 
    private BansheeCollection collection;

    public BansheeScopeProxy ()
    {
      base ();

      scope = new Unity.Scope ("/com/canonical/unity/scope/banshee");
      scope.search_in_global = true;
      scope.activate_uri.connect (activate);
      scope.sources.add_option ("banshee", _("Banshee"), null);

      base.initialize ();

      try {
	collection = new BansheeCollection ();
      } catch (DatabaseError error) {
	printerr ("Failed to open the Banshee collection database\n");
	return;
      }
    }

    protected override int num_results_without_search { get {return 100; } }
    protected override int num_results_global_search { get { return 20; } }
    protected override int num_results_lens_search { get { return 50; } }

    /**
     * Tells banshee to play the selected uri(s)
     */
    public Unity.ActivationResponse activate (string uri)
    {
      string[] exec = {"banshee", "--play-enqueued"};

      try {
        if (Uri.parse_scheme (uri) == "album")
          {
            debug (@"searching for tracks for $uri");
            string[] split = uri.split ("/");
            string artist = split[2];
            string title = split[3];
	    
            Album album = new Album ();
            album.artist = artist;
            album.title = title;
            // FIXME there must be a better way..
            foreach (string track in collection.get_track_uris (album))
              exec += track;
          }
        else
          {
            exec += uri;
          }
	
        exec += null;
	
        debug (@"Spawning banshee '%s'", string.joinv (" ", exec));
        Process.spawn_async (null, 
			     exec,
			     null, 
			     SpawnFlags.SEARCH_PATH,
			     null, 
			     null);
	
        return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);
      } catch (SpawnError error) {
        warning ("Failed to launch URI %s", uri);
        return new Unity.ActivationResponse (Unity.HandledType.NOT_HANDLED);
      }
    }

    public override async void perform_search (LensSearch search,
                                               SearchType search_type, 
                                               owned List<FilterParser> filters,
                                               int max_results = -1,
                                               Cancellable? cancellable = null)
    {
      int category_override = -1;
      if (search_type == SearchType.GLOBAL)
      {
        category_override = Category.MUSIC;
        // the lens shouldn't display anything for empty search
        if (is_search_empty (search)) return;
      }

      collection.search (search, search_type, filters,
                         max_results, category_override);
    }
  }
}
