# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Ubuntu One Bin."""

from twisted.internet import defer

from ubuntuone.controlpanel.gui.qt import ubuntuonebin as gui
from ubuntuone.controlpanel.gui.qt.tests import (
    BaseTestCase,
    CrashyBackend,
    CrashyBackendException,
)

# Attribute 'yyy' defined outside __init__, access to a protected member
# pylint: disable=W0201, W0212


class UbuntuOneBinTestCase(BaseTestCase):
    """Test the Ubuntu One base widget."""

    class_ui = gui.UbuntuOneBin

    def test_is_not_processing_after_creation(self):
        """After creation, is not processing."""
        self.assertFalse(self.ui.is_processing)
        self.assertFalse(self.ui.overlay.isVisible())

    def test_animation_is_active(self):
        """The animation is active."""
        self.assertFalse(self.ui.overlay.isVisible())

    def test_is_enabled_if_not_processing(self):
        """If not processing, the UI is enabled."""
        self.ui.show()  # need to show to test widgets visibility
        self.addCleanup(self.ui.hide)

        self.ui.is_processing = False

        self.assertTrue(self.ui.isEnabled())
        self.assertFalse(self.ui.overlay.isVisible())

    def test_is_not_enabled_if_processing(self):
        """If processing, the UI is disabled."""
        self.ui.show()  # need to show to test widgets visibility
        self.addCleanup(self.ui.hide)

        self.ui.is_processing = True

        self.assertFalse(self.ui.isEnabled())
        self.assertTrue(self.ui.overlay.isVisible())

    @defer.inlineCallbacks
    def test_backend_error_is_handled(self):
        """Any error from the backend is properly handled."""
        self.patch(self.ui, 'backend', CrashyBackend())
        yield self.ui.load()

        self.assertFalse(self.ui.is_processing)
        if self.memento:
            logged = self.memento.check_exception(CrashyBackendException)
            self.assertTrue(logged)
