NAME

    DateTime::Locale - Localization support for DateTime.pm

VERSION

    version 0.46

SYNOPSIS

      use DateTime::Locale;
    
      my $loc = DateTime::Locale->load('en_GB');
    
      print $loc->native_locale_name(),   "\n",
            $loc->datetime_format_long(), "\n";
    
      # but mostly just things like ...
    
      my $dt = DateTime->now( locale => 'fr' );
      print "Aujourd'hui le mois est " . $dt->month_name(), "\n";

DESCRIPTION

    DateTime::Locale is primarily a factory for the various locale
    subclasses. It also provides some functions for getting information on
    all the available locales.

    If you want to know what methods are available for locale objects, then
    please read the DateTime::Locale::Base documentation.

USAGE

    This module provides the following class methods:

 DateTime::Locale->load( $locale_id | $locale_name | $alias )

    Returns the locale object for the specified locale id, name, or alias -
    see the DateTime::Locale::Catalog documentation for a list of built in
    names and ids. The name provided may be either the English or native
    name.

    If the requested locale is not found, a fallback search takes place to
    find a suitable replacement.

    The fallback search order is:

      {language}_{script}_{territory}
      {language}_{script}
      {language}_{territory}_{variant}
      {language}_{territory}
      {language}

    Eg. For locale es_XX_UNKNOWN the fallback search would be:

      es_XX_UNKNOWN   # Fails - no such locale
      es_XX           # Fails - no such locale
      es              # Found - the es locale is returned as the
                      # closest match to the requested id

    Eg. For locale es_Latn_XX the fallback search would be:

      es_Latn_XX      # Fails - no such locale
      es_Latn         # Fails - no such locale
      es_XX           # Fails - no such locale
      es              # Found - the es locale is returned as the
                      # closest match to the requested id

    If no suitable replacement is found, then an exception is thrown.

    Please note that if you provide an id to this method, then the returned
    locale object's id() method will always return the value you gave, even
    if that value was an alias to some other id.

    This is done for forwards compatibility, in case something that is
    currently an alias becomes a unique locale in the future.

    This means that the value of $locale->id() and the object's class may
    not match.

    The loaded locale is cached, so that locale objects may be singletons.
    Calling DateTime::Locale->register(), DateTime::Locale->add_aliases(),
    or DateTime::Locale->remove_alias() clears the cache.

 DateTime::Locale->ids()

      my @ids = DateTime::Locale->ids();
      my $ids = DateTime::Locale->ids();

    Returns an unsorted list of the available locale ids, or an array
    reference if called in a scalar context. This list does not include
    aliases.

 DateTime::Locale->names()

      my @names = DateTime::Locale->names();
      my $names = DateTime::Locale->names();

    Returns an unsorted list of the available locale names in English, or
    an array reference if called in a scalar context.

 DateTime::Locale->native_names()

      my @names = DateTime::Locale->native_names();
      my $names = DateTime::Locale->native_names();

    Returns an unsorted list of the available locale names in their native
    language, or an array reference if called in a scalar context. All
    native names are utf8 encoded.

    NB: Some locales are only partially translated, so their native locale
    names may still contain some English.

 DateTime::Locale->add_aliases ( $alias1 => $id1, $alias2 => $id2, ... )

    Adds an alias to an existing locale id. This allows a locale to be
    loaded by its alias rather than id or name. Multiple aliases are
    allowed.

    If the passed locale id is neither registered nor listed in
    DateTime::Local::Catalog's list of ids, an exception is thrown.

     DateTime::Locale->add_aliases( LastResort => 'es_ES' );
    
     # Equivalent to DateTime::Locale->load('es_ES');
     DateTime::Locale->load('LastResort');

    You can also pass a hash reference to this method.

     DateTime::Locale->add_aliases( { Default     => 'en_GB',
                                      Alternative => 'en_US',
                                      LastResort  => 'es_ES' } );

 DateTime::Locale->remove_alias( $alias )

    Removes a locale id alias, and returns true if the specified alias
    actually existed.

     DateTime::Locale->add_aliases( LastResort => 'es_ES' );
    
     # Equivalent to DateTime::Locale->load('es_ES');
     DateTime::Locale->load('LastResort');
    
     DateTime::Locale->remove_alias('LastResort');
    
     # Throws an exception, 'LastResort' no longer exists
     DateTime::Locale->load('LastResort');

 DateTime::Locale->register( { ... }, { ... } )

    This method allows you to register custom locales with the module. A
    single locale is specified as a hash, and you may register multiple
    locales at once by passing an array of hash references.

    Until registered, custom locales cannot be instantiated via load() and
    will not be returned by querying methods such as ids() or names().

     register( id           => $locale_id,
               en_language  => ..., # something like 'English' or 'Afar',
    
               # All other keys are optional. These are:
               en_script    => ...,
               en_territory => ...,
               en_variant   => ...,
    
               native_language  => ...,
               native_sript     => ...,
               native_territory => ...,
               native_variant   => ...,
    
               # Optional - defaults to DateTime::Locale::$locale_id
               class   => $class_name,
    
               replace => $boolean
             )

    The locale id and English name are required, and the following formats
    should used wherever possible:

     id:   languageId[_script][_territoryId[_variantId]]
    
     Where:  languageId = Lower case ISO 639 code -
             Always choose 639-1 over 639-2 where possible.
    
     script = Title Case ISO 15924 script code
    
     territoryId = Upper case ISO 3166 code -
                   Always choose 3166-1 over 3166-2 where possible.
    
     variantId = Upper case variant id -
                 Basically anything you want, since this is typically the
                 component that uniquely identifies a custom locale.

    You cannot not use '@' or '=' in locale ids - these are reserved for
    future use. The underscore (_) is the component separator, and should
    not be used for any other purpose.

    If the "native_*" components are supplied, they must be utf8 encoded.

    If omitted, the native name is assumed to be identical to the English
    name.

    If class is supplied, it must be the full module name of your custom
    locale. If omitted, the locale module is assumed to be a
    DateTime::Locale subclass.

    Examples:

     DateTime::Locale->register
         ( id           => 'en_GB_RIDAS',
           en_language  => 'English',
           en_territory => 'United Kingdom',
           en_variant   => 'Ridas Custom Locale',
         );
    
     # Returns instance of class DateTime::Locale::en_GB_RIDAS
     my $l = DateTime::Locale->load('en_GB_RIDAS');
    
     DateTime::Locale->register
         ( id               => 'hu_HU',
           en_language      => 'Hungarian',
           en_territory     => Hungary',
           native_language  => 'Magyar',
           native_territory => 'Magyarország',
         );
    
     # Returns instance of class DateTime::Locale::hu_HU
     my $l = DateTime::Locale->load('hu_HU');
    
     DateTime::Locale->register
         ( id    => 'en_GB_RIDAS',
           name  => 'English United Kingdom Ridas custom locale',
           class => 'Ridas::Locales::CustomGB',
         );
    
     # Returns instance of class Ridas::Locales::CustomGB
     my $l = DateTime::Locale->load('en_GB_RIDAS');

    If you register a locale for an id that is already registered, the
    "replace" parameter must be true or an exception will be thrown.

    The complete name for a registered locale is generated by joining
    together the language, territory, and variant components with a single
    space.

    This means that in the first example, the complete English and native
    names for the locale would be "English United Kingdom Ridas Custom
    Locale", and in the second example the complete English name is
    "Hungarian Hungary", while the complete native name is "Magyar
    Magyarország". The locale will be loadable by these complete names
    (English and native), via the load() method.

ADDING CUSTOM LOCALES

    These are added in one of two ways:

      1. Subclass an existing locale implementing only the changes you
      require.

      2. Create a completely new locale as a new class.

    In either case the locale MUST be registered before use.

 Subclassing an existing locale

    The following example sublasses the United Kingdom English locale to
    change some the full date and time formats.

      package Ridas::Locale::en_GB_RIDAS1;
    
      use strict;
      use DateTime::Locale::en_GB;
    
      use base 'DateTime::Locale::en_GB';
    
      sub date_format_full   { 'EEEE d MMMM y' }
    
      sub time_format_full   { 'HH mm zzzz' }
    
      1;

    Now register it:

     DateTime::Locale->register
         ( id    => 'en_GB_RIDAS1',
    
           # name, territory, and variant as described in register() documentation
    
           class => 'Ridas::Locale::en_GB_RIDAS1',
         );

 Creating a completely new locale

    You are, of course, free to subclass DateTime::Locale::Base if you want
    to, though this is not required.

    Remember to register your custom locale!

    Of course, you can always do the registration in the module itself, and
    simply load it before using it.

    A completely new custom locale, one which does not subclass
    DateTime::Locale::Base, must implement a number of methods.

    The following methods can be used to get information about the locale's
    id and name.

      * $locale->id()

      The complete locale id, something like "en_US".

      * $locale->language_id()

      The language portion of the id, like "en".

      * $locale->script_id()

      The script portion of the id, like "Hant".

      * $locale->territory_id()

      The territory portion of the id, like "US".

      * $locale->variant_id()

      The variant portion of the id, like "PREEURO".

      * $locale->name()

      The locale's complete name, which always includes at least a language
      component, plus optional territory and variant components. Something
      like "English United States". The value returned will always be in
      English.

      * $locale->language()

      * $locale->script()

      * $locale->territory()

      * $locale->variant()

      The relevant component from the locale's complete name, like
      "English" or "United States".

      * $locale->native_name()

      The locale's complete name in localized form as a UTF-8 string.

      * $locale->native_language()

      * $locale->native_script()

      * $locale->native_territory()

      * $locale->native_variant()

      The relevant component from the locale's complete native name as a
      UTF-8 string.

    The following methods all return an array reference containing the
    specified data.

    The methods with "format" in the name should return strings that can be
    used a part of a string, like "the month of July". The stand alone
    values are for use in things like calendars, and the narrow form may
    not be unique (for example, in day column heading for a calendar it's
    okay to have "T" for both Tuesday and Thursday).

    The wide name should always be the full name of thing in question. The
    narrow name should be just one or two characters.

      * $locale->month_format_wide()

      * $locale->month_format_abbreviated()

      * $locale->month_format_narrow()

      * $locale->month_stand_alone_wide()

      * $locale->month_stand_alone_abbreviated()

      * $locale->month_stand_alone_narrow()

      * $locale->day_format_wide()

      * $locale->day_format_abbreviated()

      * $locale->day_format_narrow()

      * $locale->day_stand_alone_wide()

      * $locale->day_stand_alone_abbreviated()

      * $locale->day_stand_alone_narrow()

      * $locale->quarter_format_wide()

      * $locale->quarter_format_abbreviated()

      * $locale->quarter_format_narrow()

      * $locale->quarter_stand_alone_wide()

      * $locale->quarter_stand_alone_abbreviated()

      * $locale->quarter_stand_alone_narrow()

      * $locale->am_pm_abbreviated()

      * $locale->era_wide()

      * $locale->era_abbreviated()

      * $locale->era_narrow()

    The following methods return strings appropriate for the
    DateTime->format_cldr() method:

      * $locale->date_format_full()

      * $locale->date_format_long()

      * $locale->date_format_medium()

      * $locale->date_format_short()

      * $locale->date_format_default()

      * $locale->time_format_full()

      * $locale->time_format_long()

      * $locale->time_format_medium()

      * $locale->time_format_short()

      * $locale->time_format_default()

      * $locale->datetime_format_full()

      * $locale->datetime_format_long()

      * $locale->datetime_format_medium()

      * $locale->datetime_format_short()

      * $locale->datetime_format_default()

    A locale may also offer one or more formats for displaying part of a
    datetime, such as the year and month, or hour and minute.

      * $locale->format_for($name)

      These are accessed by passing a name to $locale->format_for(...),
      where the name is a CLDR-style format specifier.

      The return value is a string suitable for passing to
      $dt->format_cldr(), so you can do something like this:

        print $dt->format_cldr( $dt->locale()->format_for('MMMdd') )

      which for the "en" locale would print out something like "08 Jul".

      Note that the localization may also include additional text specific
      to the locale. For example, the "MMMMd" format for the "zh" locale
      includes the Chinese characters for "day" (日) and month (月), so you
      get something like "8月23日".

      * $locale->available_formats()

      This should return a list of all the format names that could be
      passed to $locale->format_for().

    The following methods deal with the default format lengths:

      * $locale->default_date_format_length()

      * $locale->default_time_format_length()

      These methods return one of "full", "long", "medium", or "short",
      indicating the current default format length.

      The default when an object is created is determined by the CLDR
      locale data.

      * $locale->set_default_date_format_length($length)

      * $locale->set_default_time_format_length($length)

      These methods accept one of "full", "long", "medium", or "short",
      indicating the new default format length.

    There are also some miscellaneous methods locales should support:

      * $locale->prefers_24_hour_time()

      Returns a boolean indicating whether or not the locale prefers
      24-hour time.

      * $locale->first_day_of_week()

      Returns a number from 1 to 7 indicating the local first day of the
      week, with Monday being 1 and Sunday being 7.

SUPPORT

    Please be aware that all locale data has been generated from the CLDR
    (Common Locale Data Repository) project locales data). The data is
    incomplete, and will contain errors in some locales.

    When reporting errors in data, please check the primary data sources
    first, then where necessary report errors directly to the primary
    source via the CLDR bug report system. See
    http://unicode.org/cldr/filing_bug_reports.html for details.

    Once these errors have been confirmed, please forward the error report
    and corrections to the DateTime mailing list, datetime@perl.org.

    Support for this module is provided via the datetime@perl.org email
    list. See http://lists.perl.org/ for more details.

DONATIONS

    If you'd like to thank me for the work I've done on this module, please
    consider making a "donation" to me via PayPal. I spend a lot of free
    time creating free software, and would appreciate any support you'd
    care to offer.

    Please note that I am not suggesting that you must do this in order for
    me to continue working on this particular software. I will continue to
    do so, inasmuch as I have in the past, for as long as it interests me.

    Similarly, a donation made in this way will probably not make me work
    on this software much more, unless I get so many donations that I can
    consider working on free software full time, which seems unlikely at
    best.

    To donate, log into PayPal and send money to autarch@urth.org or use
    the button on this page: http://www.urth.org/~autarch/fs-donation.html

SEE ALSO

    DateTime::Locale::Base

    datetime@perl.org mailing list

    http://datetime.perl.org/

AUTHORS

      * Richard Evans

      * Dave Rolsky <autarch@urth.org>

COPYRIGHT AND LICENSE

    This software is copyright (c) 2015 by Dave Rolsky.

    This is free software; you can redistribute it and/or modify it under
    the same terms as the Perl 5 programming language system itself.

