/*
 * Copyright (C) 2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VanDine <ken.vandine@canonical.com>
 */

using Dee;
using Gee;
using Config;
using Unity;
using Gwibber;

namespace UnityGwibber {

  /* DBus name for the place. Must match out .place file */
  const string BUS_NAME = "com.canonical.Unity.Lens.Gwibber";
  const string ICON_PATH = Config.PKGDATADIR + "/ui/icons/hicolor/scalable/places/";

  /**
   * The Daemon class implements all of the logic for the place.
   *
   */
  public class Daemon : GLib.Object
  {
    private Unity.Lens lens;
    private Unity.Scope scope;
    private Gwibber.Streams streams_service;
    private Gwibber.Service service;
    private Gwibber.Accounts accounts;
    private Gwibber.Utils utils;
    private Dee.Model? _model;
    private Dee.Model? _streams_model;
    private Dee.Filter _sort_filter;
    /* Keep track of the previous search, so we can determine when to
     * filter down the result set instead of rebuilding it */
    private unowned Dee.ModelIter _stream_iter_first = null;
    private unowned Dee.ModelIter _stream_iter_last = null;

    private Dee.Analyzer _analyzer;
    private Dee.Index _index;
    private Dee.ICUTermFilter _ascii_filter;

    construct
    {
      streams_service = new Gwibber.Streams();
      service = new Gwibber.Service();
      utils = new Gwibber.Utils();
      accounts = new Gwibber.Accounts();

      scope = new Unity.Scope ("/com/canonical/unity/scope/gwibber");
      scope.search_in_global = true;

      lens = new Unity.Lens("/com/canonical/unity/lens/gwibber", "gwibber");
      lens.search_in_global = true;
      lens.search_hint = _("Enter name or content you would like to search for");
      lens.visible = true;
      populate_categories ();
      populate_filters();
      lens.add_local_scope (scope);

      /* Listen for filter changes */
      scope.notify["active"].connect(() =>
      {
        if (scope.active)
        {
          scope.queue_search_changed (SearchType.DEFAULT);
        }
      });

      scope.generate_search_key.connect ((lens_search) =>
      {
        return lens_search.search_string.strip ();
      });
      scope.search_changed.connect ((lens_search, search_type, cancellable) =>
      {
        if (search_type == SearchType.DEFAULT)
          update_scope_search.begin (lens_search, cancellable);
        else
          update_global_search.begin (lens_search, cancellable);
      });

      lens.notify["active"].connect ((obj, pspec) => {
        if (lens.active && scope.active)
        {
          if (_stream_iter_first != _model.get_first_iter () || _stream_iter_last != _model.get_last_iter ())
          {
            if (scope.active)
            {
              scope.queue_search_changed (SearchType.DEFAULT);
            }
          }
        }
      });

      scope.filters_changed.connect (() =>
      {
        scope.queue_search_changed (SearchType.DEFAULT);
      });

      try
      {
        lens.export ();
      } catch (GLib.IOError e)
      {
        warning ("failed to export lens: %s", e.message);
      }

      _streams_model = streams_service.stream_model;
      Intl.setlocale(LocaleCategory.COLLATE, "C");
      _sort_filter = Dee.Filter.new_collator_desc (StreamModelColumn.TIMESTAMP);
      _model = new Dee.FilterModel (_streams_model, _sort_filter);

      _ascii_filter = new Dee.ICUTermFilter.ascii_folder ();
      _analyzer = new Dee.TextAnalyzer ();
      _analyzer.add_term_filter ((terms_in, terms_out) =>
      {
        for (uint i = 0; i < terms_in.num_terms (); i++)
        {
          unowned string term = terms_in.get_term (i);
          var folded = _ascii_filter.apply (term);
          terms_out.add_term (term);
          if (folded != term) terms_out.add_term (folded);
        }
      });
      var reader = Dee.ModelReader.new ((model, iter) =>
      {
        var sender_col = StreamModelColumn.SENDER;
        var msg_col = StreamModelColumn.MESSAGE;
        return "%s\n%s".printf (model.get_string (iter, sender_col), 
                                model.get_string (iter, msg_col));
      });
      _index = new Dee.TreeIndex (_model, _analyzer, reader);
    }

    private void populate_filters ()
    {
      var filters = new GLib.List<Unity.Filter> ();

      /* Stream filter */
      {
        var filter = new CheckOptionFilter ("stream", _("Stream"));

        filter.add_option ("messages", _("Messages"));
        filter.add_option ("replies", _("Replies"));
        filter.add_option ("images", _("Images"));
        filter.add_option ("videos", _("Videos"));
        filter.add_option ("links", _("Links"));
        filter.add_option ("private", _("Private"));
        filter.add_option ("public", _("Public"));

        filters.append (filter);
      }

      /* Account filter */
      {
        var filter = create_account_filter ();
        filters.append (filter);
      }

      lens.filters = filters;

      accounts.created.connect((source) => {
        /* FIXME: we need a way to add an option 
        var filter = scope.get_filter ("account_id") as CheckOptionFilter;
        filter.add_option (source.id, source.service + "/" + source.username);
        */
      });

      accounts.deleted.connect(() => {
        /* FIXME: we need a way to remove an option or remove and re-add the filter 
        var filter = scope.get_filter ("account_id") as CheckOptionFilter;
        lens.filters.remove (filter);
        filter = null;
        filter = create_account_filter ();
        lens.filters.append (filter);
        */
      });
    }

    private Unity.CheckOptionFilter create_account_filter ()
    {
      GLib.Icon icon;
      GLib.File icon_file;
      var filter = new CheckOptionFilter ("account_id", _("Account"));
      foreach (var _acct in accounts.list ())
      {
        /* create a service icon, which isn't used yet as of libunity 4.24
           when it is used, we can drop _acct.service from the option name 
           and rely on the service icon and username to disambiguate accounts */
        icon_file = GLib.File.new_for_path (GLib.Path.build_filename (Config.PKGDATADIR + "/plugins/" + _acct.service + "/ui/icons/16x16/" + _acct.service + ".png"));
        icon = new GLib.FileIcon (icon_file);

        if (_acct.name != null)
          filter.add_option (_acct.id, _acct.name, icon);
        else
          filter.add_option (_acct.id, _acct.service + "/" + _acct.username, icon);
      }
      return filter;
    }


    private void populate_categories ()
    {
      var categories = new GLib.List<Unity.Category> ();
      Icon icon;

      var icon_dir = File.new_for_path (ICON_PATH);

      icon = new FileIcon (icon_dir.get_child ("group-messages.svg"));
      var cat = new Unity.Category (_("Messages"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-replies.svg"));
      cat =  new Unity.Category (_("Replies"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-images.svg"));
      cat =  new Unity.Category (_("Images"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-videos.svg"));
      cat =  new Unity.Category (_("Videos"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-links.svg"));
      cat =  new Unity.Category (_("Links"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-private.svg"));
      cat =  new Unity.Category (_("Private"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-public.svg"));
      cat =  new Unity.Category (_("Public"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      lens.categories = categories;
    }


    private bool is_empty_search (LensSearch search)
    {
      return search.search_string.strip () == "";
    }

    private async void update_global_search (LensSearch search, Cancellable cancellable)
    {
      var results_model = scope.global_results_model;

      // FIXME: no results for home screen of the dash?
      if (is_empty_search (search))
      {
        search.finished ();
        return;
      }

      update_results_model (results_model, search.search_string, null);

      search.finished ();
    }

    private async void update_scope_search  (LensSearch search, Cancellable cancellable)
    {
      var results_model = search.results_model;

      update_results_model (results_model, search.search_string, null);
      message ("%u results", results_model.get_n_rows ());
      search.finished ();
    }


    /* Generic method to update a results model. We do it like this to minimize
     * code dup between updating the global- and the entry results model */
    private void update_results_model (Dee.Model results_model,
                                       string? search, Categories? category)
    {
      unowned Dee.ModelIter iter, end;

      var stream_ids = new Gee.ArrayList<string> ();
      var filter = scope.get_filter("stream") as CheckOptionFilter;
      if (filter.filtering)
      {
        foreach (Unity.FilterOption option in filter.options)
        {
          if (option.active)
          {
            stream_ids.add (option.id);
          }
        }
      }

      var account_ids = new Gee.ArrayList<string> ();
      filter = scope.get_filter("account_id") as CheckOptionFilter;
      if (filter.filtering)
      {
        foreach (Unity.FilterOption option in filter.options)
        {
          if (option.active)
          {
            account_ids.add (option.id);
          }
        }
      }

      results_model.clear ();

      iter = _model.get_first_iter ();
      end = _model.get_last_iter ();

      _stream_iter_first = _model.get_first_iter ();
      _stream_iter_last = end;
     
      var term_list = Object.new (typeof (Dee.TermList)) as Dee.TermList;
      // search only the folded terms, FIXME: is that a good idea?
      _analyzer.tokenize (_ascii_filter.apply (search), term_list);

      var matches = new Sequence<Dee.ModelIter> ();
      for (uint i = 0; i < term_list.num_terms (); i++)
      {
        // FIXME: use PREFIX search only for the last term?
        var result_set = _index.lookup (term_list.get_term (i),
                                        Dee.TermMatchFlag.PREFIX);
        bool first_pass = i == 0;
        CompareDataFunc<Dee.ModelIter> cmp_func = (a, b) =>
        {
          return a == b ? 0 : ((void*) a > (void*) b ? 1 : -1);
        };
        // intersect the results (cause we want to AND the terms)
        var remaining = new Sequence<Dee.ModelIter> ();
        foreach (var item in result_set)
        {
          if (first_pass)
            matches.insert_sorted (item, cmp_func);
          else if (matches.lookup (item, cmp_func) != null)
            remaining.insert_sorted (item, cmp_func);
        }
        if (!first_pass) matches = (owned) remaining;
        // final result set empty already?
        if (matches.get_begin_iter () == matches.get_end_iter ()) break;
      }

      matches.sort ((a, b) =>
      {
        var col = StreamModelColumn.TIMESTAMP;
        return _model.get_string (b, col).collate (_model.get_string (a, col));
      });

      var match_iter = matches.get_begin_iter ();
      var match_end_iter = matches.get_end_iter ();
      while (match_iter != match_end_iter)
      {
        iter = match_iter.get ();

        if (matches_filters (_model, iter, stream_ids, account_ids))
        {
          add_result (_model, iter, results_model);
        }

        match_iter = match_iter.next ();
      }

      if (term_list.num_terms () > 0) return;

      /* Go over the whole model if we had empty search */
      while (iter != end)
      {
        if (matches_filters (_model, iter, stream_ids, account_ids))
        {
          add_result (_model, iter, results_model);
        }
        iter = _model.next (iter);
      }

      //debug ("Results has %u rows", results_model.get_n_rows());
    }

    private bool matches_filters (Dee.Model model, Dee.ModelIter iter,
                                  Gee.List<string> stream_ids,
                                  Gee.List<string> account_ids)
    {
      bool stream_match = true;
      bool account_match = true;
      if (stream_ids.size > 0)
      {
        stream_match = model.get_string (iter, StreamModelColumn.STREAM) in stream_ids;
      }
      if (account_ids.size > 0)
      {
        string[] _accounts_array = (string[])model.get_value (iter, StreamModelColumn.ACCOUNTS);
        foreach (var a in _accounts_array)
        {
          string _account = a.split(":")[0];
          if (!(_account in account_ids))
            account_match = false;
        }
      }
      return account_match && stream_match;
    }

    private void add_result (Dee.Model model, Dee.ModelIter iter, Dee.Model results_model)
    {
      Categories group = Categories.MESSAGES;

      unowned string stream_id = 
        model.get_string (iter, StreamModelColumn.STREAM);
      switch (stream_id)
      {
        case "messages": group = Categories.MESSAGES; break;
        case "replies": group = Categories.REPLIES; break;
        case "images": group = Categories.IMAGES; break;
        case "videos": group = Categories.VIDEOS; break;
        case "links": group = Categories.LINKS; break;
        case "private": group = Categories.PRIVATE; break;
        case "public": group = Categories.PUBLIC; break;
      }
      
      string _icon_uri = model.get_string (iter, StreamModelColumn.ICON_URI);
      var _avatar_cache_image = utils.avatar_path (_icon_uri);
      if (_avatar_cache_image == null)
      {
        try
        {
          _avatar_cache_image = service.avatar_path (_icon_uri);
        } catch (GLib.Error e)
        {
        }
        if (_avatar_cache_image == null)
          _avatar_cache_image = _icon_uri;
      }

      results_model.append (model.get_string(iter, StreamModelColumn.URL),
                            _avatar_cache_image,
                            group,
                            "text/html",
                            _model.get_string(iter, StreamModelColumn.SENDER),
                            _model.get_string(iter, StreamModelColumn.MESSAGE));
    }

  } /* End Daemon class */
} /* end Gwibber namespace */
