/*
 * Copyright (C) 2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <njpatel@gmail.com>
 */

namespace TabWidgets
{
  public class Entry : Gtk.Entry
  {
    public Entry ()
    {

    }
    
    construct
    {
      //get_style_context ().add_class ("menubar");
      set_has_frame (false);
      set_size_request (-1, 24);

      Gdk.RGBA color;
      color = get_style_context ().get_background_color (Gtk.StateFlags.NORMAL);
      color.red = 1.0f;
      color.green = 0.0f;
      color.blue = 0.0f;
      color.alpha = 0.0f;
      override_background_color (Gtk.StateFlags.NORMAL, color);

      set ("placeholder-text", "Search...");
    }
  }

  public class ToggleButton : Gtk.Button
  {
    private Gtk.Label _label;
    private bool _active;

    public string text { get; construct set; }

    public bool active {
      get { return _active; }
      set
      {
        _active = value;
        toggled (active);
        queue_draw ();
      }
    }

    public signal void toggled (bool active);

    public ToggleButton (string text)
    {
      Object (text:text);
    }

    construct
    {
      _label = new Gtk.Label (text);
      add (_label);

      clicked.connect (() =>
      {
        _active = !_active;
        toggled (_active);
        queue_draw ();
      });
    }

    public override bool draw (Cairo.Context cr)
    {
      if (_active)
      {
        cr.set_line_width (1.0);
        cr.rectangle (1, 1, 5, 5);
        cr.fill ();
      }
      propagate_draw (_label, cr);
      return false;
    }
  }

  public class RadioButton : Gtk.RadioButton
  {
    private Gtk.Label _label;

    public string text { get; construct set; }

    public RadioButton (string text)
    {
      Object (text:text);
    }

    construct
    {
      _label = new Gtk.Label (text);
      _label.set_alignment (0.5f, 0.5f);
      add (_label);
    }

    public override bool draw (Cairo.Context cr)
    {
      if (active)
      {
        cr.translate (-1, -1);
        Gtk.Allocation a;
        get_allocation (out a);

        Gdk.RGBA c;
        c = get_style_context ().get_color (Gtk.StateFlags.NORMAL);

        var pat = new Cairo.Pattern.linear (0, a.height -1, a.width, a.height -1);
        pat.add_color_stop_rgba (0.0f, c.red, c.green, c.blue, 0.0f);
        pat.add_color_stop_rgba (0.5f, c.red, c.green, c.blue, 0.5f);
        pat.add_color_stop_rgba (1.0f, c.red, c.green, c.blue, 0.0f);
        cr.set_source (pat);
        cr.rectangle (0, a.height - 1, a.width, 1);
        cr.fill ();
      }
      propagate_draw (_label, cr);
      return false;
    }
  }

  public class Button : Gtk.Button
  {
    private Gtk.Label _label;

    public string text { get; construct set; }

    public Button (string text)
    {
      Object (text:text);
    }

    construct
    {
      _label = new Gtk.Label (text);
      _label.set_alignment (0.5f, 0.5f);
      add (_label);
    }

    public override bool draw (Cairo.Context cr)
    {
      propagate_draw (_label, cr);
      return false;
    }
  }

  public class AccountRadioButton : Gtk.RadioButton
  {
    public string color {  get; construct;}
    private bool _active;

    public new bool active {
      get { return _active; }
      set
      {
        _active = value;
        toggled ();
        queue_draw ();
      }
    }

    public AccountRadioButton (string color)
    {
      Object (color:color);
    }

    construct
    {
      set_size_request (24, 24);
      clicked.connect (() =>
      {
        _active = true;
        toggled ();
        queue_draw ();
      });
    }

    public override bool draw (Cairo.Context context) 
    {
      double factor = 0.3;

      if (active)
      {
        context.translate(-1, -1);
        factor = 1.0;
      } else {
        factor = 0.3;
      }

      context.push_group ();

      Gtk.Allocation a;
      get_allocation (out a);
            
      Gdk.RGBA c = Gdk.RGBA ();
      c.red = 1.0f;
      c.green = 1.0f;
      c.blue = 1.0f;
      c.alpha = 1.0f;

      c.parse (color);
      context.set_source_rgba (c.red, c.green, c.blue, 1.0f);

      context.rectangle (a.width - 2, a.height - 6, 2, 6);
      context.rectangle (a.width - 6, a.height - 2, 6, 2);
      context.fill ();
            
      propagate_draw (get_child (), context);
                        
      context.pop_group_to_source ();
      context.paint_with_alpha (factor);

      return true;
    }
  }

  public class AccountSelectorBar : Gtk.Box
  {
    public Gwibber.Accounts accounts_service;
    public Gee.HashMap<string,AccountRadioButton> accounts_buttons_map;
    private Gtk.Box abox;
    private string _selected = null;
    private AccountRadioButton account_button;

    public AccountSelectorBar ()
    {
      Object (spacing:0);
    }

    public string selected {
      get { return _selected; }
      set
        {
          if (_selected != value);
          {
            if (value == null)
            {
              foreach (var k in accounts_buttons_map.keys)
              {
                _selected = k;
                continue;
              }
            } else
              _selected = value;
            if (_selected != null)
            {
              foreach (var k in accounts_buttons_map.keys)
              {
                if (_selected != k)
                  accounts_buttons_map[k].active = false;
                else
                  accounts_buttons_map[k].active = true;
                accounts_buttons_map[_selected].show_all ();
              }
            } else 
              show_all ();
          }
        }
    }

    construct
    {
      accounts_service = new Gwibber.Accounts ();
      // Add buttons to button area at the bottom
      var box = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 12);

      accounts_buttons_map = new Gee.HashMap<string,AccountRadioButton> ();

      var accounts_list = accounts_service.list ();
            
      abox = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 6);
      abox.set_homogeneous (true);
      box.add (abox);

      for (int i = 0; i < accounts_list.length(); i++)
      {
        Gwibber.Account account = accounts_list.nth_data(i);
        if (account == null) {
          continue;
        }
        if (account.service != "twitter" && account.service != "identica" && account.service != "statusnet")
        {
          debug ("%s not supported", account.service);
          continue;
        }
        account_button = create_button (account);
        account_button.join_group (account_button);
        if (i == 0)
        {
          selected = account.id;
          account_button.active = true;
        } else
          account_button.active = false;
        
        accounts_buttons_map[account.id] = account_button;

        account_button.clicked.connect((but) => {
          if (account_button.active)
          {
            foreach (var k in accounts_buttons_map.keys)
            {
              var b = accounts_buttons_map[k];
              if (b == but)
              {
                selected = k;
              }
            }
          }
        });
      }

      add(box);
      accounts_service.created.connect((account) => {
        if (account.service != "twitter" && account.service != "identica" && account.service != "statusnet")
        {
          debug ("%s not supported", account.service);
          return;
        }
        account_button = create_button (account);
        accounts_buttons_map[account.id] = account_button;
        account_button.clicked.connect((but) => {
          if (account_button.active)
          {
            foreach (var k in accounts_buttons_map.keys)
            {
              var b = accounts_buttons_map[k];
              if (b == but)
              {
                selected = k;
              }
            }
          }
        });
      });
      accounts_service.deleted.connect((id) => {
        accounts_buttons_map[id].destroy ();
        accounts_buttons_map.unset (id);
        if (selected == id)
          selected = null;
      });
    }

    private AccountRadioButton create_button (Gwibber.Account account)
    { 
      Gtk.Image icon_service = new Gtk.Image.from_icon_name(account.service, Gtk.IconSize.MENU);
  
      icon_service.show();
      AccountRadioButton account_button = new AccountRadioButton(account.color);
      account_button.tooltip_text = (account.service + " (" + account.username + ")");

      account_button.set_image(icon_service);
      

      //account_button.clicked.connect((source) => {
      //  account_button.set_active(!account_button.get_active ());
      //});

      abox.add (account_button);
      account_button.show_all ();
      return account_button;
    }
  }
}
