"""testing desktopcouch/local_files.py module"""

import os
import shutil
import tempfile

from mocker import MockerTestCase

import desktopcouch
import desktopcouch.application.tests as test_environment

from desktopcouch.application.tests import TestCase
from desktopcouch.application.local_files import Context

# pylint: disable=W0212


class TestLocalFiles(TestCase):
    """Testing that local files returns the right things"""

    def setUp(self):
        super(TestLocalFiles, self).setUp()
        cf = test_environment.test_context.configuration
        # Test loading from file.
        cf._fill_from_file(test_environment.test_context.file_ini)
        self.xdg_path = os.environ.get("XDG_CONFIG_DIRS")
        os.environ["XDG_CONFIG_DIRS"] = os.path.join(
                os.path.dirname(__file__), os.pardir, os.pardir,
                os.pardir, "config")

    def tearDown(self):
        super(TestLocalFiles, self).tearDown()
        if self.xdg_path is None:
            del os.environ["XDG_CONFIG_DIRS"]
        else:
            os.environ["XDG_CONFIG_DIRS"] = self.xdg_path

    def test_all_files_returned(self):
        """Check local_files lists all the files that it needs to."""
        # pylint: disable=W0621,W0612
        import desktopcouch.application.local_files
        # pylint: enable=W0621,W0612
        for required in [
                "file_log", "file_ini", "file_pid", "file_stdout",
                "file_stderr", "db_dir"]:
            #"couch_exe", "couch_exec_command"
            self.assertTrue(required in dir(test_environment.test_context))

    def test_xdg_overwrite_works(self):
        """Test whether xdg_overwrite works."""
        # XXX: this should really check that it's in os.environ["TMP"]
        self.assertTrue(
            test_environment.test_context.file_ini.startswith("/tmp"))

    def test_couch_chain_ini_files(self):
        "Is compulsory-auth.ini picked up by the ini file finder?"
        # pylint: disable=W0621,W0612
        import desktopcouch.application.local_files
        # pylint: enable=W0621,W0612
        ok = [
            x for x in
            test_environment.test_context.couch_chain_ini_files().split()
            if x.endswith("compulsory-auth.ini")]
        self.assertTrue(len(ok) > 0)

    def test_bind_address(self):
        """Test bind_address."""
        old = desktopcouch.application.local_files.get_bind_address(
                ctx=test_environment.test_context)
        octets = old.split(".")
        octets[2] = str((int(octets[2]) + 128) % 256)
        new = ".".join(octets)
        desktopcouch.application.local_files.set_bind_address(new,
                ctx=test_environment.test_context)

        self.assertEquals(
            desktopcouch.application.local_files.get_bind_address(
                    ctx=test_environment.test_context),
            new)
        desktopcouch.application.local_files.set_bind_address(old,
                ctx=test_environment.test_context)
        self.assertEquals(
            desktopcouch.application.local_files.get_bind_address(
                    ctx=test_environment.test_context),
            old)


class TestKeyringIntegration(MockerTestCase):
    """Keyring integration test case."""

    def setUp(self):
        super(TestKeyringIntegration, self).setUp()
        self._keyring = self.mocker.mock()
        self.tempdir = tempfile.mkdtemp()
        self.addCleanup(shutil.rmtree, self.tempdir)
        self.cache_dir = os.path.join(self.tempdir, 'cache')
        self.data_dir = os.path.join(self.tempdir, 'data')
        self.config_dir = os.path.join(self.tempdir, 'config')

    def test_with_auth(self):
        """Test with authentication."""
        # assert that the credentials will be retrieved from the
        # keyring when we require oauth
        self._keyring.get_user_name_password()
        self.mocker.result(('user', 'password'))
        self._keyring.get_oauth_data()
        self.mocker.result(
            ('consumer_key', 'consumer_secret', 'token', 'token_secret'))
        self.mocker.replay()
        Context(self.cache_dir, self.data_dir,
            self.config_dir, with_auth=True, keyring=self._keyring)

    def test_with_no_auth(self):
        """Test without authentication."""
        Context(self.cache_dir, self.data_dir, self.config_dir,
            with_auth=False, keyring=self._keyring)
