# Copyright 2011 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Author: Manuel de la Pena <manuel.delapena@canonical.com>
"""Base dir implementation on windows."""

import logging
import os
import sys

SHELL_FOLDERS_KEY = \
    r'Software\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders'


# error W0702 is disable so that the tests can be ran on linux
# pylint: disable=W0702
def _get_special_folders(registry=None):
    """Grab all the windows shell folder locations from the registry."""
    if not registry:
        # pylint: disable=F0401
        import _winreg as registry
        # pylint: enable=F0401
    result = {}
    try:
        hive = registry.ConnectRegistry(None, registry.HKEY_CURRENT_USER)
    except:
        logging.warn('Cannot connect to registry hive HKEY_CURRENT_USER.')
        return {}
    # open the registry key where Windows stores the shell folder locations
    try:
        key = registry.OpenKey(hive, SHELL_FOLDERS_KEY)
    except:
        registry.CloseKey(hive)
        logging.warn('Cannot open registry key %s', SHELL_FOLDERS_KEY)
        return {}
    # go though the shell folder and add them to the result
    try:
        for folder_index in range(0, registry.QueryInfoKey(key)[1]):
            folder_data = registry.EnumValue(key, folder_index)
            result[folder_data[0]] = folder_data[1]
    except:
        # do not return a partial result, log error and return empty dict
        logging.warn('Error when retrieving shell folders info')
        result = {}
    finally:
        # close used resource
        registry.CloseKey(key)
        registry.CloseKey(hive)
    return result


def _assert_not_illegal_chars(resource):
    """Assert that the resource has not illegal windows chars."""
    for illegal in '<>:"/\\|?*':
        assert illegal not in resource

# ensure that we are running on windows
if sys.platform == 'win32':
    SPECIAL_FOLDERS = _get_special_folders()

    # point the cache and the config folders to the correct location
    CONFIG_HOME = os.environ.get('XDG_CONFIG_HOME',
        SPECIAL_FOLDERS['Local AppData'])
    CACHE_HOME = os.environ.get('XDG_CACHE_HOME',
        os.path.join(SPECIAL_FOLDERS['Local AppData'], 'Temp'))
    DATA_HOME = os.environ.get('XDG_DATA_HOME',
        SPECIAL_FOLDERS['Local AppData'])


def save_config_path(resource):
    """$XDG_CONFIG_HOME/<resource>/ exists, and return its path."""
    _assert_not_illegal_chars(resource)
    path = os.path.join(CONFIG_HOME, resource)
    if not os.path.isdir(path):
        os.makedirs(path, 0700)
    return path


def save_data_path(resource):
    """Ensure $XDG_DATA_HOME/<resource>/ exists, and return its path."""
    _assert_not_illegal_chars(resource)
    path = os.path.join(DATA_HOME, resource)
    if not os.path.isdir(path):
        os.makedirs(path)
    return path


def load_config_paths(ctx):
    """Load config desktopcouch config path and given ctx."""
    yield ctx.config_dir
    for config_dir in os.environ.get('XDG_CONFIG_DIRS',
        os.path.join(CONFIG_HOME, 'xdg')).split(':'):
        path = os.path.join(config_dir, "desktop-couch")
        if os.path.exists(path):
            yield path
