/**************************************************************************************
 * Copyright (c) Jonas Bonr, Alexandre Vasseur. All rights reserved.                 *
 * http://aspectwerkz.codehaus.org                                                    *
 * ---------------------------------------------------------------------------------- *
 * The software in this package is published under the terms of the LGPL license      *
 * a copy of which has been included with this distribution in the license.txt file.  *
 **************************************************************************************/
package org.codehaus.aspectwerkz.reflect.impl.asm;

import org.codehaus.aspectwerkz.reflect.ClassInfo;
import org.codehaus.aspectwerkz.reflect.FieldInfo;
import org.codehaus.aspectwerkz.transform.inlining.AsmHelper;
import org.codehaus.aspectwerkz.annotation.instrumentation.asm.AsmAnnotationHelper;
import org.codehaus.aspectwerkz.proxy.ProxyCompiler;

import org.objectweb.asm.Type;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.attrs.Attributes;

import java.util.List;
import java.util.ArrayList;
import java.io.IOException;
import java.io.InputStream;

/**
 * ASM implementation of the FieldInfo interface.
 *
 * @author <a href="mailto:jboner@codehaus.org">Jonas Bonr </a>
 */
public class AsmFieldInfo extends AsmMemberInfo implements FieldInfo {

    /**
     * The field type name.
     */
    private String m_typeName;

    /**
     * The field type.
     */
    private ClassInfo m_type = null;

    /**
     * Creates a new field java instance.
     *
     * @param field
     * @param declaringType
     * @param loader
     */
    AsmFieldInfo(final FieldStruct field, final String declaringType, final ClassLoader loader) {
        super(field, declaringType, loader);
        m_typeName = Type.getType(field.desc).getClassName();
    }

    /**
     * Returns the field info for the field specified.
     *
     * @param fieldName
     * @param fieldDesc
     * @param bytecode
     * @param loader
     * @return the field info
     */
    public static FieldInfo getFieldInfo(final String fieldName,
                                         final String fieldDesc,
                                         final byte[] bytecode,
                                         final ClassLoader loader) {
        String className = AsmClassInfo.retrieveClassNameFromBytecode(bytecode);
        AsmClassInfoRepository repository = AsmClassInfoRepository.getRepository(loader);
        ClassInfo classInfo = repository.getClassInfo(className);
        if (classInfo == null) {
            classInfo = AsmClassInfo.getClassInfo(bytecode, loader);
        }
        return classInfo.getField(AsmHelper.calculateFieldHash(fieldName, fieldDesc));
    }

    /**
     * Returns the signature for the element.
     *
     * @return the signature for the element
     */
    public String getSignature() {
        return AsmHelper.getFieldDescriptor(this);
    }

    /**
     * Returns the type.
     *
     * @return the type
     */
    public ClassInfo getType() {
        if (m_type == null) {
            m_type = AsmClassInfo.getClassInfo(m_typeName, (ClassLoader) m_loaderRef.get());
        }
        return m_type;
    }

    /**
     * Returns the annotations.
     *
     * @return the annotations
     */
    public List getAnnotations() {
        if (m_annotations == null) {
            try {
                InputStream in = null;
                ClassReader cr = null;
                try {
                    if ((ClassLoader) m_loaderRef.get() != null) {
                        in = ((ClassLoader) m_loaderRef.get()).getResourceAsStream(
                                m_declaringTypeName.replace('.', '/') + ".class"
                        );
                    } else {
                        in = ClassLoader.getSystemClassLoader().getResourceAsStream(
                                m_declaringTypeName.replace('.', '/') + ".class"
                        );
                    }
                    if (in == null) {
                        in = ProxyCompiler.getProxyResourceAsStream((ClassLoader) m_loaderRef.get(), m_declaringTypeName);
                    }
                    cr = new ClassReader(in);
                } finally {
                    try {
                        in.close();
                    } catch (Exception e) {
                        ;
                    }
                }
                List annotations = new ArrayList();
                cr.accept(
                        new AsmAnnotationHelper.FieldAnnotationExtractor(
                                annotations, m_member.name, (ClassLoader) m_loaderRef.get()
                        ),
                        Attributes.getDefaultAttributes(),
                        true
                );
                m_annotations = annotations;
            } catch (IOException e) {
                // unlikely to occur since ClassInfo relies on getResourceAsStream
                System.err.println(
                        "WARN - could not load " + m_declaringTypeName + " as a resource to retrieve annotations"
                );
                m_annotations = AsmClassInfo.EMPTY_LIST;
            }
        }
        return m_annotations;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof FieldInfo)) {
            return false;
        }
        FieldInfo fieldInfo = (FieldInfo) o;
        if (!m_declaringTypeName.equals(fieldInfo.getDeclaringType().getName())) {
            return false;
        }
        if (!m_member.name.equals(fieldInfo.getName())) {
            return false;
        }
        if (!m_typeName.equals(fieldInfo.getType().getName())) {
            return false;
        }
        return true;
    }

    public int hashCode() {
        int result = 29;
        result = (29 * result) + m_declaringTypeName.hashCode();
        result = (29 * result) + m_member.name.hashCode();
        result = (29 * result) + m_typeName.hashCode();
        return result;
    }

    public String toString() {
        StringBuffer sb = new StringBuffer(m_declaringTypeName);
        sb.append('.').append(m_member.name).append(' ');
        sb.append(m_member.desc);
        return sb.toString();
    }
}